# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from logging import getLogger, Logger  # noqa: UnusedImport

from common.models.geo import CityMajority, Settlement
from common.utils.geobase import geobase
from contextlib2 import contextmanager
from travel.rasp.wizards.wizard_lib.geobase_region_type import GeobaseRegionType, LikeSettlementTypesIds


class RegionCapitalProvider(object):
    def __init__(self, geobase, logger):
        # type: (Geobase, Logger) -> None
        self._cache = None
        self._geobase = geobase
        self._logger = logger

    @contextmanager
    def using_precache(self):
        if not self.is_precached():
            self.build_cache()
            try:
                yield
            finally:
                self.clean()
        else:
            yield

    def is_precached(self):
        return self._cache is not None

    def clean(self):
        self._cache = None

    def build_cache(self):
        region_capitals = Settlement.objects.filter(
            majority__lte=CityMajority.REGION_CAPITAL_ID,
            region_id__isnull=False,
            hidden=False
        )

        cache = {}
        for capital in region_capitals:
            cache[capital.region_id] = capital
        self._cache = cache
        self._logger.info("Precache %d region capitals", len(self._cache))

    def find_region_capital(self, geo_id):
        # type: (int) -> Optional[Settlement]
        if not geo_id:
            self._logger.debug("Can not find the region capital, because geo id is None")
            return None

        geo_object, geo_object_type = self._find_by_geo_id(geo_id)
        if geo_object is None:
            self._logger.debug(
                "Can not find the region capital, because geobase does not know about it. geo_id: %d",
                geo_id
            )
            return None

        if geo_object_type not in LikeSettlementTypesIds:
            self._logger.debug(
                "Can not find the region capital, because geo object has wrong type. geo_id: %d geo_object_type: %s",
                geo_id, geo_object_type
            )
            return None

        parent_settlement = self._find_parent_with_type(
            geo_object=geo_object,
            expected_region_type=GeobaseRegionType.CITY,
        )

        # some region -> some fake region capital -> some real city
        if parent_settlement is not None:
            try:
                return Settlement.hidden_manager.get(_geo_id=parent_settlement.id)
            except (ValueError, Settlement.DoesNotExist):
                pass

        region = self._find_parent_with_type(
            geo_object=geo_object,
            expected_region_type=GeobaseRegionType.REGION,
        )
        if region is None:
            self._logger.debug(
                "Can not find the region capital, because geobase does not geo object region. geo_id: %d",
                geo_id
            )
            return None

        region_capital = self._cache.get(region.id)
        if region_capital is None:
            self._logger.debug(
                "Can not find the region capital, because rasp base does not know it or"
                "settlement is hidden. geo_id: %d region_id: %d",
                geo_id, region.id
            )

            return None
        self._logger.info(
            "Can find region capital. geo_id:"
            "geo_object_id: %d, geo_object_title: %s, region_capital_id %s, region_capital_title: %s",
            geo_object.id, geo_object.name.decode("utf-8"), region_capital.id, region_capital.title
        )
        return region_capital

    def _find_parent_with_type(self, geo_object, expected_region_type):
        parent_id = geo_object.parent_id
        while parent_id != 0:
            region, region_type = self._find_by_geo_id(parent_id)
            if region is None:
                return None
            if region_type == expected_region_type:
                return region
            parent_id = region.parent_id
        return None

    def _find_by_geo_id(self, geo_id):
        try:
            geo_object = self._geobase.region_by_id(geo_id)
        except RuntimeError:
            return None, None

        try:
            geo_object_type = GeobaseRegionType(geo_object.type)
        except ValueError:
            return None, None
        return geo_object, geo_object_type


region_capital_provider = RegionCapitalProvider(
    geobase=geobase,
    logger=getLogger(__name__)
)
