# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from django.core.exceptions import ObjectDoesNotExist
from rest_framework import serializers

from common.models.geo import Point, Settlement
from travel.rasp.wizards.wizard_lib.region_capital_provider import region_capital_provider


REGION2SETTLEMENT = {977: 146, 10933: 18, 10231: 11319, 11398: 78, 11282: 64, 11443: 74, 11010: 28}


def dump_point(point):
    return {'key': point.point_key, 'title': point.L_popular_title()} if point else None


def parse_point_key(query_params, point_key_name, allow_blank=False):
    point_key = query_params.get(point_key_name)
    if point_key:
        try:
            return Point.get_by_key(point_key)
        except ValueError:
            raise serializers.ValidationError('invalid point key {}={!r}'.format(point_key_name, point_key))
        except ObjectDoesNotExist:
            raise serializers.ValidationError('unknown point key {}={!r}'.format(point_key_name, point_key))
    elif not allow_blank:
        raise serializers.ValidationError('{} have to be specified'.format(point_key_name))


def _parse_settlement_geoid(query_params, settlement_geoid_name):
    raw_settlement_geoid = query_params.get(settlement_geoid_name)
    if not raw_settlement_geoid:
        return None
    try:
        settlement_geoid = int(raw_settlement_geoid)
    except ValueError:
        raise serializers.ValidationError('invalid geoid {}={!r}'.format(settlement_geoid_name, raw_settlement_geoid))

    if settlement_geoid in REGION2SETTLEMENT:
        settlement_geoid = REGION2SETTLEMENT[settlement_geoid]
    try:
        return Settlement.hidden_manager.get(_geo_id=settlement_geoid)
    except (ValueError, Settlement.DoesNotExist):
        region_capital = region_capital_provider.find_region_capital(settlement_geoid)
        if region_capital:
            return region_capital
        raise serializers.ValidationError('unknown geoid {}={!r}'.format(settlement_geoid_name, raw_settlement_geoid))


def parse_point(query_params, point_key_name, settlement_geoid_name, allow_blank=False):
    point_by_key = parse_point_key(
        query_params=query_params,
        point_key_name=point_key_name,
        allow_blank=True,
    )
    settlement_by_geoid = _parse_settlement_geoid(
        query_params=query_params,
        settlement_geoid_name=settlement_geoid_name,
    )

    if bool(point_by_key) ^ bool(settlement_by_geoid):
        return point_by_key or settlement_by_geoid
    elif point_by_key or not allow_blank:
        raise serializers.ValidationError('one of arguments have to be specified: {} or {}'
                                          .format(point_key_name, settlement_geoid_name))


def parse_settlement_geoid(query_params, settlement_geoid_name):
    try:
        return _parse_settlement_geoid(query_params, settlement_geoid_name)
    except serializers.ValidationError:
        pass
