# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import calendar
from travel.library.python.tracing.instrumentation import traced_function

from django.utils.encoding import force_bytes
from django.utils.http import urlencode

from common.models.currency import Currency
from travel.rasp.wizards.wizard_lib.protobuf_models.direction_response_pb2 import TDirectionResponse
from travel.rasp.wizards.wizard_lib.protobuf_models.thread_pb2 import TThread
from travel.rasp.wizards.wizard_lib.views.helpers.service_urls import format_morda_url, format_touch_url


def _update_timestamp(proto_timestamp, local_dt):
    proto_timestamp.Timestamp = calendar.timegm(local_dt.utctimetuple())
    proto_timestamp.UtcOffset = int(local_dt.utcoffset().total_seconds())


def _update_thread(proto_thread, segment, language):
    proto_thread.Number = segment.train_number
    proto_thread.StartDate = segment.thread_start_date.toordinal()
    proto_thread.Title = getattr(segment.train_title, language)

    if segment.thread_transport_subtype_id is not None:
        proto_thread.TransportSubtypeId = segment.thread_transport_subtype_id

    thread_express_type = segment.thread_express_type
    if thread_express_type is not None:
        proto_thread.ExpressType = TThread.EExpressType.Value(thread_express_type.name)


def _update_price(proto_price, price):
    if not price:
        return

    proto_price.Value = price.value
    proto_price.CurrencyId = Currency.objects.get(code=price.currency).id


def _update_segment_urls(proto_urls, segment, query):
    url_path = b'/thread/{}/'.format(force_bytes(segment.thread_uid))
    parameters = [
        ('departure_from', segment.departure_local_dt.strftime('%Y-%m-%d %H:%M:%S')),
        ('point_from', query.departure_point.point_key),
        ('point_to', query.arrival_point.point_key),
        ('station_from', segment.departure_station.id),
        ('station_to', segment.arrival_station.id),
    ]
    if query.main_reqid is not None:
        parameters.append(('wizardReqId', query.main_reqid))
    url_query = urlencode(parameters)
    proto_urls.Desktop = format_morda_url(url_path, url_query, query.tld)
    proto_urls.Mobile = format_touch_url(url_path, url_query, query.tld)


@traced_function
def dump_segments(segments, query):
    proto_segments = TDirectionResponse(Total=len(segments))

    for segment in segments[:query.limit]:
        proto_segment = proto_segments.Segments.add(
            DepartureStationId=segment.departure_station.id, ArrivalStationId=segment.arrival_station.id
        )

        _update_timestamp(proto_segment.DepartureTimestamp, segment.departure_local_dt)
        _update_timestamp(proto_segment.ArrivalTimestamp, segment.arrival_local_dt)
        _update_thread(proto_segment.Thread, segment, query.language)
        _update_price(proto_segment.Price, segment.price)

        _update_segment_urls(proto_segment.Urls, segment, query)

    return proto_segments
