# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import date, timedelta

import six
from django.core.exceptions import ObjectDoesNotExist
from django.template.loader import select_template
from django.utils import translation
from jinja2 import Environment
from singledispatch import singledispatch

from common.models.geo import Station
from common.models.currency import Currency, Price
from common.utils.date import human_date, human_duration
from common.xgettext.i18n import stringify


@singledispatch
def humanize_filter(value):
    return six.text_type(value)


@humanize_filter.register(date)
def _humanize_date(date_value):
    return human_date(date_value)


@humanize_filter.register(timedelta)
def _humanize_timedelta(timedelta_value):
    return human_duration(timedelta_value)


@humanize_filter.register(Price)
def _humanize_price(price, **kwargs):
    try:
        currency = Currency.objects.get(code=price.currency)
    except ObjectDoesNotExist:
        currency = Currency.objects.get(iso_code=price.currency)
    return currency.format_value(price.value, **kwargs)


@humanize_filter.register(Station)
def _humanize_station(station, genitive=False, capitalized=False, with_settlement=False):
    local_case = 'genitive' if genitive else 'nominative'

    popular_title = station.L_popular_title(fallback=False, case=local_case)
    if popular_title:
        station_title = popular_title
    else:
        station_type_name = station.station_type.L_name(case=local_case)
        station_title = translation.ugettext('station_with_type') % {
            'station': station.L_title(),
            'type': station_type_name.capitalize() if capitalized else station_type_name,
        }

    if with_settlement:
        settlement = station.settlement
        if settlement:
            settlement_title = settlement.L_title()
            if settlement_title != station.L_title():
                return translation.ugettext('station_with_settlement') % {
                    'settlement': settlement_title,
                    'station': station_title,
                }

    return station_title


def human_list_filter(items, lang=None):
    lang = lang or translation.get_language()
    human_list_func = ru_human_list_ili if lang == 'ru' else punctuate_content
    return stringify(human_list_func(list(items)))


def get_environment(**options):
    env = Environment(**options)
    env.filters.update(humanize=humanize_filter, human_list=human_list_filter)

    if 'jinja2.ext.i18n' in options.get('extensions', []):
        env.install_gettext_translations(translation, newstyle=True)

    return env


def render_macros(template_name, context, macro_names, experiments):
    template_names = ['{}.{}.j2'.format(template_name, e.value) for e in experiments]
    template_names += ['{}.j2'.format(template_name)]
    template = select_template(template_names).template

    module = template.make_module(context)
    return {
        macro_name: six.text_type(getattr(module, macro_name)())
        for macro_name in macro_names
        if hasattr(module, macro_name)
    }


# TODO: cannot import bemhmtl because of import hemi
#  from common.utils.bemhtml import punctuate_content, ru_human_list_ili
from six.moves import filter as ifilter


def punctuate_content(content, sep=', '):
    punctuated_content = []

    iterator = ifilter(None, content)

    try:
        item = next(iterator)
    except StopIteration:
        return []

    while True:
        punctuated_content.append(item)

        try:
            item = next(iterator)
        except StopIteration:
            break

        punctuated_content.append(sep)

    return punctuated_content


def ru_human_list_ili(l):
    if len(l) < 2:
        return l

    return punctuate_content(l[:-1], sep=', ') + [' или ', l[-1]]
