# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from collections import namedtuple
from itertools import islice, starmap

from django.utils.functional import cached_property


def chunker(iterable, chunk_size):
    iterator = iter(iterable)
    while True:
        chunk = list(islice(iterator, chunk_size))
        if chunk:
            yield chunk
        else:
            break


class ChunkedSizedMap(object):

    def __init__(self, func, queryset, pk_index,
                 chunk_size=100000, object_ids=None):
        self.func = func
        self.queryset = queryset
        self.pk_index = pk_index
        self.chunk_size = chunk_size
        self.object_ids = object_ids

    @cached_property
    def _pks(self):
        pks = self.queryset.values_list('pk', flat=True)

        if self.object_ids is None:
            return list(pks)

        return [
            pk
            for ids_part in chunker(self.object_ids, self.chunk_size)
            for pk in pks.filter(id__in=ids_part)
        ]

    def __iter__(self):
        new_qs = self.queryset.order_by()

        pk_index = self.pk_index
        func = self.func
        for pks_part in chunker(self._pks, self.chunk_size):
            objects = {
                obj[pk_index]: obj
                for obj in new_qs.filter(id__in=pks_part)
            }

            for pk in pks_part:
                yield func(*objects[pk])

    def __len__(self):
        return len(self._pks)


class OriginalBoxField(object):

    def __init__(self, name, box):
        self.name = name
        self.box = box


_factory_template = '''
def factory({factory_args}, *_):
    return {make_call}
'''


class OriginalBox(object):

    def __init__(self, *fields, **options):
        instance_mixins = options.pop('instance_mixins', None)

        if options:
            raise TypeError('Unexpected options {!r}'.format(options))

        self.fields = fields

        tuple_fields = []
        query_fields = []

        for field in fields:
            if isinstance(field, basestring):
                tuple_fields.append(field)
                query_fields.append(field)
            else:
                tuple_fields.append(field.name)
                query_fields.extend(
                    '{}__{}'.format(field.name, query_field)
                    for query_field in field.box.query_fields
                )

        tuple_factory = namedtuple('_originalbox', tuple_fields)
        if instance_mixins is not None:
            tuple_factory = type(str('_originalbox'), (tuple_factory,) + tuple(instance_mixins), {})

        self.instance_factory = tuple_factory
        self.query_fields = query_fields

    def __ror__(self, name):
        return OriginalBoxField(name, self)

    def _get_factory_args(self, subpath=None):
        factory_args = []
        make_args = []
        namespace = {}
        arg_prefix = '{}__'.format(subpath) if subpath else ''

        for field in self.fields:
            if isinstance(field, basestring):
                factory_args.append(arg_prefix + field)
                make_args.append(arg_prefix + field)
            else:
                field_subpath = arg_prefix + field.name
                field_factory_args, field_make_call, field_namespace = \
                    field.box._get_factory_args(field_subpath)

                factory_args.extend(field_factory_args)
                make_args.append(field_make_call)
                namespace.update(field_namespace)

        make_name = '_make_{}'.format(subpath) if subpath else '_make'
        namespace[make_name] = self.instance_factory
        make_call = '{}({})'.format(make_name, ', '.join(make_args))

        return factory_args, make_call, namespace

    def _make_factory(self, verbose=False):
        factory_args, make_call, namespace = self._get_factory_args()

        factory_definition = _factory_template.format(
            factory_args=', '.join(factory_args),
            make_call=make_call
        )

        if verbose:
            print(factory_definition)

        exec factory_definition in namespace

        return namespace['factory']

    def iter_queryset(self, queryset):
        return starmap(
            self._make_factory(),
            queryset.values_list(*self.query_fields)
        )

    def iter_queryset_chunked(self, queryset,
                              chunk_size=100000, object_ids=None):
        query_fields = self.query_fields

        pk_field = queryset.model._meta.pk
        pk_field_name = pk_field.db_column or pk_field.name

        try:
            pk_index = query_fields.index(pk_field_name)
        except ValueError:
            pk_index = len(query_fields)
            query_fields.append(pk_field_name)

        queryset = queryset.values_list(*query_fields)

        return ChunkedSizedMap(
            self._make_factory(), queryset, pk_index,
            chunk_size=chunk_size, object_ids=object_ids
        )
