# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import signal
from threading import RLock

from django.db import connections

from common.models.timestamp import Timestamp
from common.utils.threadutils import Ticker
from common.db.switcher import switcher

logger = logging.getLogger(__name__)

DEFAULT_INTERVAL = 10  # секунды
_MISSING = object()


class TimestampChecker(Ticker):
    _db_switched = False
    _value = _MISSING

    def __init__(self, code, callback, interval=DEFAULT_INTERVAL, logger=logger, **kwargs):
        self._code = code
        self._callback = callback
        self._logger = logger
        self._lock = RLock()
        super(TimestampChecker, self).__init__(interval=interval, **kwargs)

    def run(self):
        super(TimestampChecker, self).run()

    def on_db_switched(self, **kwargs):
        with self._lock:
            self._db_switched = True

    def _prepare_connections(self):
        if self._db_switched:
            self._logger.debug('TimestampChecker is closing database connections')
            connections.close_all()
            del self._db_switched, self._value

    def _update_value(self):
        value = Timestamp.get(self._code)
        if self._value is not _MISSING and self._value != value:
            self._callback(self._value, value)
        self._value = value

    def tick(self):
        try:
            with self._lock:
                self._prepare_connections()
                self._update_value()
        except Exception:
            self._logger.exception('Exception while checking timestamp')


def setup_timestamped_recache(logger=logger, **kwargs):
    server_pid = os.getpid()

    def recache(previous_value, current_value):
        logger.info('Timestamp changed from %s to %s, recaching', previous_value, current_value)
        os.kill(server_pid, signal.SIGHUP)

    checker = TimestampChecker(callback=recache, logger=logger, **kwargs)
    switcher.db_switched.connect(checker.on_db_switched)
    checker.start()
