# coding: utf8
from collections import defaultdict, namedtuple, deque
from datetime import timedelta

import numpy

from travel.tools.devprocess_metrics.startrek.model import Aggregate

WINDOW_SIZE = timedelta(60)

_Key = namedtuple('_Key', ['team', 'sp', 'queue', 'is_feature'])


class Aggregator:
    def __init__(self):
        self._aggregates = []
        self._window_by_key = defaultdict(deque)

    def add_issue(self, issue):
        sp = issue.sp if issue.sp <= 8 else 999
        keys = [
            _Key(issue.team, sp, issue.queue, issue.is_feature),
            _Key(issue.team, sp, 'ALL', issue.is_feature)
        ]
        for key in keys:
            window = self._window_by_key[key]
            window.append(issue)
            self._aggregates.append(self.calculate_aggregate(key, window))
            if not window or issue.closed_at - window[0].closed_at < WINDOW_SIZE:
                continue
            window.popleft()

    @staticmethod
    def calculate_aggregate(key, window):
        ttms = [i.ttm for i in window]
        ts = window[-1].closed_at
        return Aggregate(
            team=key.team,
            queue=key.queue,
            sp=key.sp,
            is_feature=key.is_feature,
            ts=ts,

            median=numpy.median(ttms),
            perc95=numpy.percentile(ttms, 95),
            perc90=numpy.percentile(ttms, 90),
            perc75=numpy.percentile(ttms, 75),
            std3=numpy.std(ttms) * 3,
        )

    def get_aggregates(self):
        # in order to account last window
        for key, window in self._window_by_key.items():
            self.calculate_aggregate(key, window)
        return self._aggregates
