# coding: utf8
import argparse

from infi.clickhouse_orm import Database
from loguru import logger
from startrek_client import Startrek

from travel.library.python.rasp_vault.api import get_secret
from travel.tools.devprocess_metrics.startrek.aggregate import Aggregator
from travel.tools.devprocess_metrics.startrek.conf import Teams, FILTERS_BY_TEAM
from travel.tools.devprocess_metrics.startrek.model import IssuesRepository, AggregateRepository, RetortaLinksRepository
from travel.tools.devprocess_metrics.startrek.issue import (
    StCollector, IssueCreator, ChildrenAwareStateTracker, NaiveStateTracker,
    estimate_from_sp, estimate_avia,
    filter_by_queue_wl, check_feature_by_tag,
)
from travel.tools.devprocess_metrics.startrek.retorta_links import save_retorta_links


def main():
    st_token = get_secret('rasp-robot-tokens-and-passwords.startrek-oauth-token')

    st_client = Startrek(useragent='v2', token=st_token)
    parser = argparse.ArgumentParser()
    parser.add_argument('--recreate', help='Whether or not we should recreate table', action="store_true")

    args = parser.parse_args()
    to_process = [
        StCollector(st_client,
                    query=FILTERS_BY_TEAM[Teams.KOMOD.value],
                    issue_creator=IssueCreator(
                        team=Teams.KOMOD.value,
                        estimate_extractor=estimate_from_sp,
                        feature_checker=check_feature_by_tag,
                        state_tracker=NaiveStateTracker())),
        StCollector(st_client,
                    query=FILTERS_BY_TEAM[Teams.AVIA.value],
                    issue_creator=IssueCreator(
                        team=Teams.AVIA.value,
                        estimate_extractor=estimate_avia,
                        feature_checker=check_feature_by_tag,
                        state_tracker=ChildrenAwareStateTracker(
                            is_suitable=filter_by_queue_wl(['RASPTICKETS', 'TRAVELFRONT'])))),
        StCollector(st_client,
                    query=FILTERS_BY_TEAM[Teams.OTT.value],
                    issue_creator=IssueCreator(
                        team=Teams.OTT.value,
                        estimate_extractor=estimate_from_sp,
                        feature_checker=check_feature_by_tag,
                        state_tracker=NaiveStateTracker())),
        # StCollector(st_client,
        #             query=FILTERS_BY_TEAM[Teams.HOTELS.value],
        #             issue_creator=IssueCreator(
        #                 team=Teams.HOTELS.value,
        #                 estimate_extractor=estimate_hotels,
        #                 feature_checker=check_feature_by_queue(['HOTELS']),
        #                 state_tracker=NaiveStateTracker())),
        StCollector(st_client,
                    query=FILTERS_BY_TEAM[Teams.TRAIN_O_BUSES.value],
                    issue_creator=IssueCreator(
                        team=Teams.TRAIN_O_BUSES.value,
                        estimate_extractor=estimate_from_sp,
                        feature_checker=check_feature_by_tag,
                        state_tracker=NaiveStateTracker())),

        # StCollector(st_client, 'Marketing', query='Queue: TRAVELITM AND Tags: feature AND resolution: fixed'),

        # StCollector(st_client, 'Suburban', query='Queue: SUBURBAN AND Tags: feature AND resolution: fixed'),
        # StCollector(st_client, 'TravelApp', query='Queue: TRAVELAPP AND Tags: feature AND resolution: fixed'),
    ]

    db = Database(
        'devmetrics',
        'https://sas-cksd4pfhlk3ecj5f.db.yandex.net:8443',
        get_secret('rasp-misc.rasp_devmetrics_ch_user'),
        get_secret('rasp-misc.rasp_devmetrics_ch_password')
    )

    aggregates_repository = AggregateRepository(db)
    issues_repository = IssuesRepository(db)
    retorta_repository = RetortaLinksRepository(db)
    if args.recreate:
        logger.info('recreating tables')
        try:
            issues_repository.recreate_table()
            aggregates_repository.recreate_table()
            retorta_repository.recreate_table()
        except Exception as ex:
            logger.exception(ex)

    save_retorta_links(retorta_repository)
    aggregator = Aggregator()
    for collector in to_process:
        issues = []
        for issue in collector.collect():
            issues.append(issue)
            aggregator.add_issue(issue)
        issues_repository.save_many(issues)

    aggregates_repository.save_many(aggregator.get_aggregates())


if __name__ == '__main__':
    main()
