# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from dataclasses import dataclass
from datetime import datetime
from operator import attrgetter
from typing import List

from dateutil.parser import parse
from loguru import logger
from pytz import UTC
from yandex_tracker_client import exceptions
from travel.tools.devprocess_metrics.startrek.model import Issue


class SkippedIssueError(Exception):
    pass


@dataclass
class _IssueState:
    work_started_at: datetime  # first transition to "in progress"
    resolved_at: datetime  # last transition to "resolved"
    duplicate: bool

    @classmethod
    def create(cls, issue):
        state = _IssueState(None, None, False)
        for entry in issue.changelog:
            if not cls._is_status_switch(entry):
                continue
            status = cls._get_status(entry)

            if status == 'duplicate':
                state.duplicate = True
            if status != 'open' and state.work_started_at is None:
                state.work_started_at = parse(entry.updatedAt)
            if status == 'closed' or status == 'fixed':
                state.resolved_at = parse(entry.updatedAt)
        return state

    @staticmethod
    def _is_status_switch(entry):
        if entry.type != 'IssueWorkflow':
            return False
        for f in entry.fields:
            if f['field'].id == 'status':
                return True
        return False

    @staticmethod
    def _get_status(entry):
        for field in entry.fields:
            if 'to' in field and hasattr(field['to'], 'key'):
                return field['to'].key
        return None


class NaiveStateTracker:
    """Naive state tracking, using only issue status.
    """

    def get_state(self, raw_issue) -> _IssueState:
        return _IssueState.create(raw_issue)


class ChildrenAwareStateTracker:
    """Using statuses of children tasks in order to determine feature state.
    """
    def __init__(self, is_suitable=None):
        self._is_suitable = is_suitable if is_suitable else lambda _: True

    def get_state(self, raw_issue) -> _IssueState:
        """
        Should work as follows:
          - work_started_at - datetime when first child task was switched "in progress" (except of design and analytics)
          - closed_at - min(datetime when feature was closed, datetime when last child task was closed (except of design and analytics))
        """
        subtask_work_started_at = UTC.localize(datetime(4000, 1, 1))
        subtask_resolved_at = UTC.localize(datetime(1900, 1, 1))
        if raw_issue.links is not None:
            for link in raw_issue.links:
                if self._is_subtask(link):
                    try:
                        # definitely not a task
                        if not hasattr(link, 'queue'):
                            continue
                        if not self._is_suitable(link):
                            continue
                        subtask_state = _IssueState.create(link.object)
                        if subtask_state.work_started_at and subtask_work_started_at > subtask_state.work_started_at:
                            subtask_work_started_at = subtask_state.work_started_at
                        if subtask_state.resolved_at and subtask_resolved_at < subtask_state.resolved_at:
                            subtask_resolved_at = subtask_state.resolved_at
                    except exceptions.Forbidden:
                        print('Error in subtask: {}'.format(link))

        state = _IssueState.create(raw_issue)
        # # hack for tasks with strange statuses for ex. RASPTICKETS-21602
        # if state.resolved_at is None:
        #     state.resolved_at = subtask_resolved_at
        state.resolved_at = max(state.resolved_at, subtask_resolved_at)
        state.work_started_at = min(state.work_started_at, subtask_work_started_at)
        return state

    @staticmethod
    def _is_subtask(link):
        return link.direction == 'outward' and link.type.id == 'subtask'


class IssueCreator:
    def __init__(self, team, estimate_extractor, feature_checker=None, state_tracker=NaiveStateTracker(), post_process=None):
        self._team = team
        self._extract_estimate = estimate_extractor
        self._is_feature = feature_checker
        self._state_tracker = state_tracker
        self._post_process = post_process

    def create_issue(self, raw_issue):
        state = self._state_tracker.get_state(raw_issue)

        ttm = (state.resolved_at - state.work_started_at).days
        issue = Issue(
            team=self._team,
            queue=raw_issue.queue.key,

            key=raw_issue.key,
            last_sprint=raw_issue.sprint[-1].name if raw_issue.sprint else '',
            sprints=list(map(attrgetter('name'), raw_issue.sprint)),
            url='https://st.yandex-team.ru/{}'.format(raw_issue.key),
            tags=raw_issue.tags,
            sp=self._extract_estimate(raw_issue),
            ttm=ttm,

            closed_at=state.resolved_at,
            duplicate=state.duplicate,

            is_feature=self._is_feature(raw_issue)
        )

        if self._post_process:
            self._post_process(issue)
        return issue


class StCollector:
    def __init__(self, st_client, issue_creator=None, filter_=None, query=None, filter_id=None):
        self._st_client = st_client
        self._issue_creator = issue_creator
        self._filter = filter_
        self._query = query
        self._filter_id = filter_id

    def collect(self):
        counter = 0
        for raw_issue in self._iterate():
            try:
                issue = self._issue_creator.create_issue(raw_issue)
                if issue.duplicate:
                    continue
                counter += 1
                yield issue
            except SkippedIssueError:
                logger.error('Issue {}: skipped'.format(raw_issue.key))
            except Exception:
                logger.exception('For issue {} - error captured'.format(raw_issue.key))
        logger.info('Collected %s issues' % counter)

    def _iterate(self):
        params = {'per_page': 1000, 'order': ['-resolvedAt']}
        if self._query:
            params['query'] = self._query
        elif self._filter_id:
            params['filter_id'] = self._filter_id
        else:
            params['filter'] = self._filter
        for raw_issue in self._st_client.issues.find(**params):
            yield raw_issue


def estimate_from_sp(raw_issue):
    sp = raw_issue.storyPoints
    if sp is not None:
        return float(sp)
    return 0


def estimate_avia(raw_issue):
    sp = raw_issue.releaseComment
    if sp in ('skipped', 'skip'):
        raise SkippedIssueError
    if sp is None:
        sp = raw_issue.storyPoints
    if sp is not None:
        return float(sp)
    return -1


def estimate_hotels(raw_issue):
    #  Queues/b'619b561b834ce9073aef8f80--spFe
    sp_fe = raw_issue.spFe or -1
    sp_be = raw_issue.spBe or -1
    return max(sp_fe, sp_be)


def filter_by_queue_wl(white_list_queues: List[str]):
    def _filter(obj):
        if obj.queue not in white_list_queues:
            return False
        return True
    return _filter


def filter_by_queue_bl(black_list_queues: List[str]):
    def _filter(obj):
        if obj.queue in black_list_queues:
            return False
        return True
    return _filter


def check_feature_by_queue(queues):
    def _check(raw_issue):
        if raw_issue.queue in queues:
            return True
        return False
    return _check


def check_feature_by_tag(raw_issue):
    if 'feature' in raw_issue.tags:
        return True
    return False


def posprocess_avia(issue: Issue):
    product_sprints = [s for s in issue.sprints if 'backend' not in s]
    if product_sprints:
        issue.sprints = product_sprints
    issue.last_sprint = product_sprints[-1] if product_sprints else 'None'
