package ru.yandex.travel.tools.dicts;

import java.io.FileOutputStream;
import java.io.IOException;
import java.util.concurrent.TimeUnit;

import com.google.common.base.Stopwatch;
import lombok.extern.slf4j.Slf4j;
import picocli.CommandLine;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

import ru.yandex.travel.dicts.avia.TAirline;
import ru.yandex.travel.dicts.avia.TAirport;
import ru.yandex.travel.dicts.avia.TSettlement;
import ru.yandex.travel.tools.dicts.model.Airline;
import ru.yandex.travel.tools.dicts.model.Airport;
import ru.yandex.travel.tools.dicts.model.Settlement;

/**
 * Usage: ya make && mkdir ./testing_dicts && ./run.sh ru.yandex.travel.tools.dicts.AviaDictsExport --backend-url=http://backend.testing.avia.yandex.net/ --export-dir=./testing_dicts
 */
@Command(description = "Downloads avia dictionaries and stores them locally in the proto format", mixinStandardHelpOptions = true, version = "1.0")
@Slf4j
public class AviaDictsExport implements Runnable {
    @Option(names = "--backend-url", required = true, description = "Avia Backend API endpoint to get dictionaries from")
    private String backendUrl;
    @Option(names = "--export-dir", required = true, description = "Where to store transformed dictionaries")
    private String exportDir;

    public static void main(String[] args) {
        CommandLine.run(new AviaDictsExport(), args);
    }

    public void run() {
        Stopwatch sw = Stopwatch.createStarted();
        try {
            exportDicts();
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            log.info("Export finished in (ms): " + sw.elapsed(TimeUnit.MILLISECONDS));
        }
    }

    private void exportDicts() throws IOException {
        try (OldRemoteAviaDictsFetcher fetcher = new OldRemoteAviaDictsFetcher(backendUrl)) {
            exportSettlements(fetcher, exportDir + "/settlements.pb.data");
            exportAirports(fetcher, exportDir + "/airports.pb.data");
            exportAirlines(fetcher, exportDir + "/airlines.pb.data");
        }
    }

    private void exportSettlements(OldRemoteAviaDictsFetcher fetcher, String exportFile) throws IOException {
        try (FileOutputStream out = new FileOutputStream(exportFile)) {
            for (Settlement settlement : fetcher.fetchSettlements()) {
                var builder = TSettlement.newBuilder()
                        .setId(settlement.getId())
                        .setTitle(settlement.getTitle())
                        .setPhraseFrom(settlement.getPhraseFrom())
                        .setPhraseTo(settlement.getPhraseTo())
                        .setPhraseIn(settlement.getPhraseIn());
                if (settlement.getGeoId() != null) {
                    builder.setGeoId(settlement.getGeoId());
                }
                if (settlement.getCountryId() != null) {
                    builder.setCountryId(settlement.getCountryId());
                }
                if (settlement.getIata() != null) {
                    builder.setIataCode(settlement.getIata());
                }
                builder.build().writeDelimitedTo(out);
            }
        }
    }

    private void exportAirports(OldRemoteAviaDictsFetcher fetcher, String exportFile) throws IOException {
        try (FileOutputStream out = new FileOutputStream(exportFile)) {
            for (Airport airport : fetcher.fetchAirports()) {
                var builder = TAirport.newBuilder()
                        .setId(airport.getId());
                if (airport.getCode() != null) {
                    builder.setCode(airport.getCode());
                }
                if (airport.getIataCode() != null) {
                    builder.setIataCode(airport.getIataCode());
                }
                if (airport.getSettlementId() != null) {
                    builder.setSettlementId(airport.getSettlementId());
                }
                if (airport.getTitle() != null) {
                    builder.setTitle(airport.getTitle());
                }
                if (airport.getTimeZone() != null) {
                    builder.setTimeZoneId(airport.getTimeZone().getID());
                }
                if (airport.getPhraseFrom() != null) {
                    builder.setPhraseFrom(airport.getPhraseFrom());
                }
                if (airport.getPhraseTo() != null) {
                    builder.setPhraseTo(airport.getPhraseTo());
                }
                if (airport.getPhraseIn() != null) {
                    builder.setPhraseIn(airport.getPhraseIn());
                }
                builder.build().writeDelimitedTo(out);
            }
        }
    }

    private void exportAirlines(OldRemoteAviaDictsFetcher fetcher, String exportFile) throws IOException {
        try (FileOutputStream out = new FileOutputStream(exportFile)) {
            for (Airline airline : fetcher.fetchAirlines()) {
                var builder = TAirline.newBuilder()
                        .setId(airline.getId())
                        .setColor(airline.getColor())
                        .setLogoSvg(airline.getLogoSvg());
                if (airline.getIata() != null) {
                    builder.setIataCode(airline.getIata());
                }
                if (airline.getSirena() != null) {
                    builder.setSirenaCode(airline.getSirena());
                }
                if (airline.getAlliance() != null) {
                    builder.setAlliance(airline.getAlliance());
                }
                if (airline.getTitle() != null) {
                    builder.setTitle(airline.getTitle());
                }
                if (airline.getUrl() != null) {
                    builder.setUrl(airline.getUrl());
                }
                builder.build().writeDelimitedTo(out);
            }
        }
    }
}
