package ru.yandex.travel.tools.dicts;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.TimeZone;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.ObjectCodec;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.google.common.base.Strings;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Dsl;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;

import ru.yandex.travel.tools.dicts.model.Airline;
import ru.yandex.travel.tools.dicts.model.Airport;
import ru.yandex.travel.tools.dicts.model.Settlement;

public class OldRemoteAviaDictsFetcher implements AutoCloseable {
    private static Duration DEFAULT_TIMEOUT = Duration.ofSeconds(60);

    private final String aviaBackendHost;
    private final AsyncHttpClient ahc;
    private final ObjectMapper objectMapper;

    public OldRemoteAviaDictsFetcher(String aviaBackendHost) {
        this.aviaBackendHost = aviaBackendHost;
        this.ahc = Dsl.asyncHttpClient();
        this.objectMapper = createJsonMapper();
    }

    private static ObjectMapper createJsonMapper() {
        SimpleModule customDeserializer = new SimpleModule();
        customDeserializer.addDeserializer(Airport.class, new AirportDeserializer());
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        mapper.registerModule(customDeserializer);
        return mapper;
    }

    public List<Settlement> fetchSettlements() {
        return fetchObjects(aviaBackendHost + "/rest/settlements", Settlement.class);
    }

    public List<Airport> fetchAirports() {
        return fetchObjects(aviaBackendHost + "/rest/airports", Airport.class);
    }

    public List<Airline> fetchAirlines() {
        return fetchObjects(aviaBackendHost + "/rest/airlines/airline_info", Airline.class);
    }

    public <T> List<T> fetchObjects(String url, Class<T> type) {
        try {
            Response response = ahc.executeRequest(new RequestBuilder()
                    .setUrl(url)
                    .setRequestTimeout((int) DEFAULT_TIMEOUT.toMillis()))
                    .get();
            return parseObjects(response, type);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private <T> List<T> parseObjects(Response response, Class<T> type) throws IOException {
        JsonNode responseNode = objectMapper.readTree(response.getResponseBodyAsStream());
        String status = responseNode.get("status").asText();
        if ("ok".equalsIgnoreCase(status)) {
            return toList(responseNode.get("data")).stream()
                    .map((n) -> nodeToValue(n, type))
                    .collect(Collectors.toList());
        } else {
            throw new RuntimeException("Got error status in response: " + status);
        }
    }

    private List<JsonNode> toList(JsonNode parent) {
        List<JsonNode> nodes = new ArrayList<>();
        for (JsonNode data : parent) {
            nodes.add(data);
        }
        return nodes;
    }

    private <T> T nodeToValue(JsonNode node, Class<T> type) {
        try {
            return objectMapper.treeToValue(node, type);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Error parsing airport", e);
        }
    }

    @Override
    public void close() throws IOException {
        ahc.close();
    }

    private static class AirportDeserializer extends StdDeserializer<Airport> {
        public AirportDeserializer() {
            this(null);
        }

        public AirportDeserializer(Class<?> vc) {
            super(vc);
        }

        @Override
        public Airport deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
            ObjectCodec oc = p.getCodec();
            JsonNode airportNode = oc.readTree(p);
            return fromNode(airportNode);
        }

        private Airport fromNode(JsonNode node) {
            Airport result = new Airport();
            result.setId(node.get("id").longValue());
            result.setCode(node.get("code").textValue());
            result.setIataCode(node.get("iataCode").textValue());
            result.setTitle(node.get("title").textValue());
            String timezoneName = node.get("timeZone").textValue();
            if (!Strings.isNullOrEmpty(timezoneName)) {
                result.setTimeZone(
                        TimeZone.getTimeZone(timezoneName)
                );
            }
            result.setPhraseFrom(node.get("phraseFrom").textValue());
            result.setPhraseTo(node.get("phraseTo").textValue());
            result.setPhraseIn(node.get("phraseIn").textValue());
            if (node.get("settlement") != null &&
                    node.get("settlement").get("id") != null ) {
                result.setSettlementId(
                        node.get("settlement").get("id").longValue()
                );
            }
            return result;
        }
    }
}
