import ast
import re
from typing import Generator

from loguru import logger

from dateutil.parser import parse
from travel.tools.release_tracker.model import Release

RELEASE_PATTERN = re.compile(r'Релиз ([\w\-_]+) \d')
MAX_PER_PAGE = 1000
DEFAULT_ORDER = ['-resolvedAt']


class StartrekRMCollector:
    def __init__(self, st_client, query: str):
        self._client = st_client
        self._query = query

    def collect(self) -> Generator[Release, None, None]:
        for issue in self._client.issues.find(query=self._query, per_page=MAX_PER_PAGE, order=DEFAULT_ORDER):
            yield from self._process_issue(issue)

    def _process_issue(self, issue):
        project_match = re.search(RELEASE_PATTERN, issue.summary)
        if not project_match:
            logger.exception('No project in issue summary %s' % issue.summary)
            return
        project = project_match.group(1)
        for e in issue.changelog:
            if e.type == 'IssueUpdated' and e.comments is not None:
                for comment in e.comments.get('added', []):
                    release_data = self._process_comment(comment.text)
                    if not release_data:
                        continue
                    created = parse(comment.updatedAt)
                    try:
                        release = Release(
                            environment=release_data['spec']['sandbox']['release_type'],
                            project=project,
                            created_at=created,
                            created_at_str=created.strftime('%Y-%d-%mT%H:%M'),
                            created_ts=created.timestamp(),
                        )
                        yield release
                    except TypeError:
                        # TODO: log here
                        logger.exception('Can\'t determine release type for %s' % release_data)
                        continue

    @staticmethod
    def _process_comment(comment):
        for line in comment.split('Ok('):
            try:
                result = ast.literal_eval(line.replace(')', '', -1))
                if isinstance(result, dict):
                    return result
            except SyntaxError:
                pass
