package main

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	dictReaders "a.yandex-team.ru/travel/library/go/dicts/base"
	"a.yandex-team.ru/travel/library/go/resourcestorage"
	"a.yandex-team.ru/travel/library/go/services/sandbox"
	"errors"
)

type DictDumper struct {
	logger          log.Logger
	sandboxClient   sandbox.Client
	messageProvider MessageProvider
	locationBuilder DictLocationBuilder
}

func NewDictDumper(
	logger log.Logger,
	sandboxClient sandbox.Client,
	messageProvider MessageProvider,
	locationBuilder DictLocationBuilder,
) *DictDumper {
	return &DictDumper{
		logger:          logger,
		sandboxClient:   sandboxClient,
		messageProvider: messageProvider,
		locationBuilder: locationBuilder,
	}
}

func (d *DictDumper) Dump() error {
	errWrapper := xerrors.NewSentinel("DictDumper.Dump")

	resource, err := d.loadSandboxResource()
	if err != nil {
		return errWrapper.Wrap(err)
	}

	dictIterator, _ := dictReaders.BuildIteratorFromBytes(resource)
	writer := resourcestorage.NewS3StorageWriter(Cfg.S3Storage, Cfg.S3AccessKey, Cfg.S3Secret)

	err = d.populateStorage(dictIterator, writer)
	if err != nil {
		return errWrapper.Wrap(err)
	}
	return nil
}

func (d DictDumper) loadSandboxResource() ([]byte, error) {
	errWrapper := xerrors.NewSentinel("DictDumper.loadSandboxResource")

	ResourceType := d.locationBuilder.GetResourceName()
	resourceInfo, err := d.sandboxClient.GetResources(sandbox.ResourceQuery{
		Limit: 1,
		Type:  ResourceType,
	})
	if err != nil {
		return nil, errWrapper.Wrap(err)
	}

	resource, err := d.sandboxClient.DownloadAllResourceBytes(resourceInfo.Items[0])
	if err != nil {
		return nil, errWrapper.Wrap(err)
	}

	d.logger.Info(
		"resource was loaded",
		log.String("type", ResourceType),
		log.Int("size", len(resource)),
	)

	return resource, nil
}

func (d *DictDumper) populateStorage(
	iterator *dictReaders.BytesIterator,
	writer resourcestorage.StorageWriter,
) error {
	errWrapper := xerrors.NewSentinel("DictDumper.populateStorage")

	s3ResourceKey := d.locationBuilder.GetS3ResourceName()
	if err := writer.CreateVersion(s3ResourceKey); err != nil {
		return errWrapper.Wrap(err)
	}

	var messageCount int
	for {
		row, err := iterator.Next()
		if errors.Is(err, dictReaders.ErrStopIteration) {
			break
		}
		if err != nil {
			return errWrapper.Wrap(err)
		}
		messageCount++

		message, err := d.messageProvider.Load(row)
		if err != nil {
			return errWrapper.Wrap(err)
		}
		err = writer.Write(message)
		if err != nil {
			return errWrapper.Wrap(err)
		}

	}
	if err := writer.Commit(); err != nil {
		return errWrapper.Wrap(err)
	}
	if err := writer.CleanOldVersions(s3ResourceKey, 1); err != nil {
		return errWrapper.Wrap(err)
	}

	d.logger.Info("resource was written to s3", log.Int("message_count", messageCount))
	return nil
}
