package main

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/sandbox_planner"
	"a.yandex-team.ru/travel/library/go/services/sandbox"
	dicts "a.yandex-team.ru/travel/proto/dicts/rasp"
	"context"
	"fmt"
	"github.com/golang/protobuf/proto"
	"net/http"
	"os"
	"time"
)

const (
	Country             = "COUNTRY"
	Currency            = "CURRENCY"
	NamedTrain          = "NAMED_TRAIN"
	Region              = "REGION"
	Route               = "ROUTE"
	Settlement          = "SETTLEMENT"
	SettlementToStation = "SETTLEMENT_TO_STATION"
	Station             = "STATION"
	StationCode         = "STATION_CODE"
	Timezone            = "TIMEZONE"
	Thread              = "THREAD"
	ThreadStation       = "THREAD_STATION"
)
const (
	ConfigPathEnv   = "CONFIG_PATH"
	LocalConfigPath = "config.yaml"
)

var DictTypes = []string{
	Country,
	Currency,
	NamedTrain,
	Region,
	Route,
	Settlement,
	SettlementToStation,
	Station,
	StationCode,
	Timezone,
	Thread,
	ThreadStation,
}

func fillDictLocationBuilders(s3DictsPrefix string) DictLocationBuilders {
	dictLocationBuilders := make(DictLocationBuilders)
	for _, dictType := range DictTypes {
		dictLocationBuilders.AddRasp(dictType, s3DictsPrefix)
	}
	return dictLocationBuilders
}

func fillMessageProviders() MessageProviders {
	messageProviders := make(MessageProviders)
	messageProviders.AddProvider(Country, func() proto.Message { return new(dicts.TCountry) })
	messageProviders.AddProvider(Currency, func() proto.Message { return new(dicts.TCurrency) })
	messageProviders.AddProvider(NamedTrain, func() proto.Message { return new(dicts.TNamedTrain) })
	messageProviders.AddProvider(Region, func() proto.Message { return new(dicts.TRegion) })
	messageProviders.AddProvider(Route, func() proto.Message { return new(dicts.TRoute) })
	messageProviders.AddProvider(Settlement, func() proto.Message { return new(dicts.TSettlement) })
	messageProviders.AddProvider(SettlementToStation, func() proto.Message { return new(dicts.TStation2Settlement) })
	messageProviders.AddProvider(Station, func() proto.Message { return new(dicts.TStation) })
	messageProviders.AddProvider(StationCode, func() proto.Message { return new(dicts.TStationCode) })
	messageProviders.AddProvider(Timezone, func() proto.Message { return new(dicts.TTimeZone) })
	messageProviders.AddProvider(Thread, func() proto.Message { return new(dicts.TThread) })
	messageProviders.AddProvider(ThreadStation, func() proto.Message { return new(dicts.TThreadStation) })
	return messageProviders
}

func main() {
	if err := sandboxplanner.ReplaceArgsFromEnv(); err != nil {
		fmt.Println("can not replace args:", err)
		return
	}

	loadConfiguration()
	logger := configureLogging()

	dictLocationBuilders := fillDictLocationBuilders(Cfg.S3DictsPrefix)
	messageProviders := fillMessageProviders()

	sandboxClient := sandbox.NewHTTPClient(
		sandbox.BaseURL,
		&http.Client{Timeout: 5 * time.Minute},
		Cfg.SandboxToken,
	)

	dumpDict := func(dictType string) {
		locationBuilder, found := dictLocationBuilders[dictType]
		if !found {
			logger.Error("locationBuilder was not found", log.String("dict_type", dictType))
			os.Exit(1)
		}

		messageProvider, found := messageProviders[dictType]
		if !found {
			logger.Error("messageProvider was not found", log.String("dict_type", dictType))
			os.Exit(1)
		}

		logger.Info("start dict dumping", log.String("dict_type", dictType))
		dumper := NewDictDumper(logger, sandboxClient, messageProvider, locationBuilder)
		err := dumper.Dump()
		if err != nil {
			logger.Error("failed to dump dict", log.Error(err), log.String("dict_type", dictType))
			os.Exit(1)
		}
	}

	if Cfg.DictType != "" {
		dumpDict(Cfg.DictType)
	} else {
		for _, dictType := range DictTypes {
			dumpDict(dictType)
		}
	}
}

func loadConfiguration() {
	if _, found := os.LookupEnv(ConfigPathEnv); !found {
		_ = os.Setenv(ConfigPathEnv, LocalConfigPath)
	}

	loader := configuration.NewDefaultConfitaLoader()
	if err := loader.Load(context.Background(), &Cfg); err != nil {
		fmt.Println("can not load configuration:", err)
		os.Exit(1)
	}
}

func configureLogging() log.Logger {
	level, _ := log.ParseLevel("info")
	return zap.Must(zap.CLIConfig(level))
}
