#!/usr/bin/env bash

set -o errexit
set -o nounset

PROGNAME=`basename $0`

: ${DEBFULLNAME:=teamcity}
: ${DEBEMAIL:=teamcity@yandex-team.ru}

PROJECT_ROOT=`pwd`
PROJECT_WORKDIR=

if [ -n "$1" ] && [ "${1:0:1}" != "-" ]; then
    PACKAGE_NAME="$1"
    PROJECT_WORKDIR="$1"
    PROJECT_DIR="${PROJECT_ROOT}/${PROJECT_WORKDIR}"
    shift
else
    PROJECT_DIR="$PROJECT_ROOT"
fi

# dry run - unconditionally build package but not deploy it
DRY_RUN=
# skip vcs-push
SKIP_VCS=
SB_TOKEN=

log() {
    echo >&2 "+ $*"
}

usage()
{
    echo \
"Please read this hints, which would help you to make love, not war

$PROGNAME [PROJECT_DIR] [OPTIONS]
    -h --help
    --email=${DEBEMAIL}
    --project-workdir= # set project dir (default=package name)
    --dry-run # unconditionally builds the package without deploying it
    --skip-vcs # skip vcs-push
    --sb-token=# Sandbox token
"
}

while [ $# -gt 0 ]; do
    PARAM=`echo $1 | cut -d = -f 1`
    VALUE=`echo $1 | cut -d = -f 2-`
    case $PARAM in
        -h | --help)
            usage
            exit 0
            ;;
        --email)
            DEBEMAIL=$VALUE
            log "I: set DEBEMAIL=$DEBEMAIL"
            ;;
        --project-workdir)
            PROJECT_WORKDIR="$VALUE"
            PROJECT_DIR="${PROJECT_ROOT}/${PROJECT_WORKDIR}"
            log "I: set PROJECT_WORKDIR=${PROJECT_WORKDIR}"
            ;;
        --dry-run)
            DRY_RUN=yes
            log "I: set DRY_RUN=${DRY_RUN}"
            ;;
        --skip-vcs)
            SKIP_VCS=yes
            log "I: set SKIP_VCS=${SKIP_VCS}"
            ;;
        --sb-token)
            SB_TOKEN=$VALUE
            log "I: set SB_TOKEN=..."
            ;;
        *)
            echo "ERROR: unknown parameter \"$PARAM\""
            usage
            exit 1
            ;;
    esac
    shift
done

init_variables() {
  pushd $PROJECT_DIR >/dev/null
  export DEB_SOURCE=$(dpkg-parsechangelog | sed -n 's/^Source: //p')
  export DEB_VERSION=$(dpkg-parsechangelog | sed -n 's/^Version: //p')
  export DEB_CHANGES=$(dpkg-parsechangelog | grep "^Changes:" -A 100 | tail -n -1)

  DEB_ARCH=$(grep "^Architecture:" debian/control | awk -F' ' '{print $NF}' | tail -n 1)
  popd >/dev/null

  if [ "$DEB_ARCH" != "all" ]; then
     log "Unexpected DEB_ARCH=${DEB_ARCH}, only 'all' is supported"
     exit 1
  fi

  CHANGES_FILE="${DEB_SOURCE}_${DEB_VERSION}_${DEB_ARCH}.changes"
  DEB_FILE="${DEB_SOURCE}_${DEB_VERSION}_${DEB_ARCH}.deb"
  PACKAGE_ID=${DEB_SOURCE}-${DEB_VERSION}


  DEBS_DIR=$PROJECT_DIR/..
}

upload_to_sandbox() {
    PROJECT_NAME=$(basename $PROJECT_DIR)
    SB_RESOURCE_TYPE=TRAVEL_DEBPACKAGE_$(echo ${PROJECT_NAME^^} | cut -d "-" -f 2)
    DESCR="Package ${PACKAGE_NAME^^} $DEB_VERSION"$'\n'"$DEB_CHANGES"
    rm -f upload.sfx.py
    curl -vOJ 'http://proxy.sandbox.yandex-team.ru/last/SANDBOX_UPLOAD_SCRIPT'
    chmod +x upload.sfx.py
    for i in {1..3}; do
        ./upload.sfx.py -T $SB_TOKEN -o TRAVEL -t $SB_RESOURCE_TYPE $DEB_FILE -I -D task-meta.json -d "$DESCR" && break
    done
    RES=$?
    if [ $RES != 0 ]; then
        echo "Failed to upload to sandbox: Retcode $RES != 0"
        exit 9
    fi
    if [ ! -f task-meta.json ]; then
        echo "Failed to upload to sandbox, no task-meta.json"
        exit 10
    fi
    SB_TASK_ID=$(jq .task.id task-meta.json)
    SB_TASK_URL=https://sandbox.yandex-team.ru/task/$SB_TASK_ID
    rm -f upload.sfx.py task-meta.json
}

debuild_cmd() {
    pushd $PROJECT_DIR >/dev/null

    debuild -rfakeroot -sa -e$DEBEMAIL -I -us -uc -b || exit 8

    log "I: debian clean..."
    debclean || exit 4

    popd >/dev/null
}


hg_push_cmd() {
    echo "push changes to vcsroot"
    hg bookmark "${PACKAGE_ID}"
    # Strange hg push return codes...
    set +e
    hg push -B "${PACKAGE_ID}"

    if [ $? -ne 1 ]; then
        exit 7
    fi
    set -e
}


# main script

echo "Init environment..."
init_variables

if [ "$DRY_RUN" == "yes"  ]; then
  echo "Dry run enabled"
  echo "Only building package..."

  debuild_cmd || exit 3

  echo "Done"
  exit 0
fi


echo "Building package..."
debuild_cmd || exit 3


echo "Uploading package to sandbox..."
upload_to_sandbox || exit 4

if [ "$SKIP_VCS" != "yes"  ]; then
    echo "Bookmarking VCS..."
    hg_push_cmd
fi

echo "##teamcity[buildNumber '${DEB_VERSION}_{build.number}']"
echo "##teamcity[buildStatus text='{build.status.text} as $SB_TASK_URL']"

echo "Done"
exit 0
