package ru.yandex.travel.yt;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import io.netty.channel.nio.NioEventLoopGroup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.yt.ytclient.bus.BusConnector;
import ru.yandex.yt.ytclient.bus.DefaultBusConnector;
import ru.yandex.yt.ytclient.proxy.ApiServiceClient;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.proxy.YtCluster;
import ru.yandex.yt.ytclient.rpc.AlwaysSwitchRpcFailoverPolicy;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;
import ru.yandex.yt.ytclient.rpc.RpcFailoverPolicy;
import ru.yandex.yt.ytclient.rpc.RpcOptions;


public class Factory {
    private static final Logger logger = LoggerFactory.getLogger(Factory.class);
    private static Duration defaultLocalTimeout = Duration.ofMillis(1000);
    private static Duration defaultGlobalTimeout = Duration.ofMillis(2000);
    private static Duration defaultPingTimeout = Duration.ofMillis(15000);
    private static Duration defaultClientTimeout = Duration.ofSeconds(5);

    private static int writeTotalRetries = 5;
    private static int writeLocalRetries = 2;
    private static RpcFailoverPolicy failoverPolicy = new AlwaysSwitchRpcFailoverPolicy();

    private int port;
    private String user;
    private String token;
    private String dc;

    private Map<String, YtClient> clusterClients = new ConcurrentHashMap<>();

    private BusConnector busConnector = null;
    private Duration localTimeout = defaultLocalTimeout;
    private Duration globalTimeout = defaultGlobalTimeout;
    private Duration pingTimeout = defaultPingTimeout;
    private Duration clientTimeout = defaultClientTimeout;


    public Factory(int port, String user, String token, String dc) {
        this.port = port;
        this.user = user;
        this.token = token;
        this.dc = dc;
        logger.info("Initialized YT factory for DC {}", dc);
        logger.info("YT user: {}", user);
    }

    public Duration getLocalTimeout() {
        return localTimeout;
    }

    public void setLocalTimeout(Duration localTimeout) {
        this.localTimeout = localTimeout;
    }

    public Duration getGlobalTimeout() {
        return globalTimeout;
    }

    public void setGlobalTimeout(Duration globalTimeout) {
        this.globalTimeout = globalTimeout;
    }

    public Duration getPingTimeout() {
        return pingTimeout;
    }

    public void setPingTimeout(Duration pingTimeout) {
        this.pingTimeout = pingTimeout;
    }

    public Duration getClientTimeout() {
        return clientTimeout;
    }

    public void setClientTimeout(Duration clientTimeout) {
        this.clientTimeout = clientTimeout;
    }

    public String getUser() {
        return user;
    }

    public String getDc() {
        return dc;
    }

    public YtClient getYtClient(List<String> clusterNames) {
        Collections.sort(clusterNames);
        String key = String.join("+", clusterNames);
        return clusterClients.computeIfAbsent(key, k -> createYtClient(clusterNames, key));
    }

    public List<ApiServiceClient> getWriteClientsForClusters(Iterable<String> clusterNames) {
        return getWriteClientsForClusters(clusterNames, writeTotalRetries, writeLocalRetries);
    }

    public List<ApiServiceClient> getWriteClientsForClusters(Iterable<String> clusterNames, int maxRetries,
                                                             int localRetries) {
        return Collections.singletonList(getYtClient((List<String>) clusterNames));
    }

    public ApiServiceClient getReadClientForClusters(Iterable<String> clusterNames) {
        return getYtClient((List<String>) clusterNames);
    }

    public void close() {
        for (YtClient client:  clusterClients.values()) {
            client.close();
        }
        busConnector.close();
    }


    private YtClient createYtClient(List<String> clusterNames, String key) {
        initConnector();
        logger.info("Creating YT client for " + key + ", cluster names: " + clusterNames);
        return new YtClient(busConnector, clusterNames.stream().map(YtCluster::new).collect(Collectors.toList()),
                dc, new RpcCredentials(user, token),
                new RpcOptions()
                        .setFailoverTimeout(localTimeout)
                        .setGlobalTimeout(globalTimeout)
                        .setPingTimeout(pingTimeout).setFailoverPolicy(failoverPolicy)
        );
    }

    private void initConnector() {
        if (busConnector == null) {
            NioEventLoopGroup g = new NioEventLoopGroup(16);
            busConnector = new DefaultBusConnector(g, true);
        }
    }

}
