package ru.yandex.travel.yt.daos;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.travel.yt.Factory;
import ru.yandex.travel.yt.ReplicationInfo;
import ru.yandex.travel.yt.YtDao;
import ru.yandex.travel.yt.mappings.TableMapping;
import ru.yandex.travel.yt.queries.QueryPart;
import ru.yandex.yt.rpcproxy.EAtomicity;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.LookupRowsRequest;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.wire.UnversionedRow;


public class SingleYtDao<T> implements YtDao<T> {
    private String basePath;
    private Factory factory;
    private String cluster;
    private final TableMapping<T> tableMapping;
    private Class<T> mappedClass;
    private Logger logger;

    private volatile CompletableFuture<ReplicationInfo> replicationInfo = null;
    private volatile CompletableFuture<Void> schemaInfo = null;
    private volatile CompletableFuture<Void> atomicityInfo = null;
    private Boolean atomic = null;
    private final Object replicationInfoSyncRoot = new Object();
    private final Object schemaInfoSyncRoot = new Object();
    private final Object atomicityInfoSyncRoot = new Object();


    private long replicaSyncTimestamp = 0;
    private Duration replicaSyncInterval = Duration.ofSeconds(10);


    public SingleYtDao(Factory factory, String cluster, String basePath, Class<T> mappedClass) {
        this(factory, cluster, basePath, mappedClass, null, null);
    }

    public SingleYtDao(Factory factory, String cluster, String basePath, Class<T> mappedClass, String tableName, Long ttl) {
        this.factory = factory;
        this.cluster = cluster;
        this.basePath = basePath;
        this.tableMapping = new TableMapping<>(mappedClass, tableName, ttl);
        this.mappedClass = mappedClass;
        this.logger = LoggerFactory.getLogger(this.getClass().getName() + "[" + mappedClass.getSimpleName() + "](" + tableMapping.getTableName() + "@" + cluster + ")");
    }

    public String getTableName() {
        return tableMapping.getTableName();
    }

    @Override
    public CompletableFuture<List<T>> select(List<QueryPart> queryParts) {
        QueryPart[] arr = new QueryPart[queryParts.size()];
        queryParts.toArray(arr);
        return getReadClient().thenCompose(client -> {
            String query = prepareQuery(arr);
            return selectInternal(client, query);
        });
    }

    @Override
    public CompletableFuture<List<T>> select(QueryPart... queryParts) {
        return getReadClient().thenCompose(client -> {
            String query = prepareQuery(queryParts);
            return selectInternal(client, query);
        });
    }

    @Override
    public CompletableFuture<Optional<T>> get(List<Object> keyValues) {
        return getReadClient().thenCompose(client -> getInternal(keyValues, client));
    }

    @Override
    public CompletableFuture<Optional<T>> get(Object... keyValues) {
        return getReadClient().thenCompose(client -> getInternal(Arrays.asList(keyValues), client));
    }

    @Override
    public CompletableFuture<Void> put(T object) {
        return getWriteClient().thenCompose(client -> putInternal(object, client));
    }

    @Override
    public CompletableFuture<Void> put(List<T> objects) {
        return getWriteClient().thenCompose(client -> putManyInternal(objects, client));
    }

    @Override
    public CompletableFuture<Void> delete(List<Object> keyValues) {
        return getWriteClient().thenCompose(client -> deleteInternal(keyValues, client));
    }

    @Override
    public CompletableFuture<Void> delete(Object... keyValues) {
        return getWriteClient().thenCompose(client -> deleteInternal(Arrays.asList(keyValues), client));
    }

    @Override
    public Class<?> getMappedClass() {
        return mappedClass;
    }

    public String getClusterName() {
        return cluster;
    }


    private String prepareQuery(QueryPart[] queryParts) {
        List<String> screenedColumns = tableMapping.getSchema().getColumnNames().stream().map(s -> "[" + s + "]").
                collect(Collectors.toList());
        String toSelect = String.join(",", screenedColumns);
        String path = basePath + "/" + tableMapping.getTableName();
        String queryTemplate = toSelect + " FROM [" + path + "]";
        if (queryParts.length > 0) {
            String condition = String.join(" AND ",
                    Arrays.stream(queryParts).map(QueryPart::getQuery).collect(Collectors.toList()));
            queryTemplate += " WHERE " + condition;
        }
        return queryTemplate;
    }


    protected String getPath() {
        return basePath + "/" + tableMapping.getTableName();
    }

    private YtClient getMasterClient() {
        logger.debug("Getting master client");
        YtClient client = factory.getYtClient(Collections.singletonList(cluster));
        int alive = client.getAliveDestinations().values().stream().mapToInt(List::size).sum();
        logger.debug("Num alive for {}: {}", cluster, alive);
        return client;
    }

    private CompletableFuture<List<T>> selectInternal(YtClient client, String query) {
        long started = System.currentTimeMillis();
        logger.debug("Selecting {}", query);
        try {
            return client.selectRows(query).thenApply(unversionedRowset -> {
                long took = System.currentTimeMillis() - started;
                logger.debug("Selected {}: {} ms", query, took);
                List<UnversionedRow> rows = unversionedRowset.getRows();
                List<T> result = new ArrayList<>(rows.size());
                for (UnversionedRow row : rows) {
                    if (row == null) {
                        continue;
                    }
                    Optional<T> r = tableMapping.fromValueList(row.getValues());
                    r.ifPresent(result::add);
                }
                return result;
            });
        } finally {
            logger.debug("Enqueuing query {} took {}", query, System.currentTimeMillis() - started);
        }
    }

    private CompletableFuture<Optional<T>> getInternal(List<Object> keys, YtClient client) {
        LookupRowsRequest request = new LookupRowsRequest(basePath + "/" + tableMapping.getTableName(),
                tableMapping.getSchema().toLookup())
                .addFilter(keys)
                .addLookupColumns(tableMapping.getSchema().getColumnNames());
        logger.debug("Looking up: {}", keys);
        long started = System.currentTimeMillis();
        return client.lookupRows(request).thenApply(unversionedRowset -> {
            long length = System.currentTimeMillis() - started;
            logger.debug("Looked up {}: {} ms", keys, length);
            List<UnversionedRow> rows = unversionedRowset.getRows();
            if (rows.size() == 0 || rows.get(0) == null) {
                return Optional.empty();
            } else {
                if (rows.size() > 1) {
                    throw new RuntimeException("Too many rows returned");
                } else {
                    return tableMapping.fromValueList(rows.get(0).getValues());
                }
            }
        });
    }

    private CompletableFuture<Void> putInternal(T object, YtClient client) {
        List<Object> values = tableMapping.toValueList(object);
        List<Object> keys = null;
        if (logger.isDebugEnabled()) {
            keys = tableMapping.toValueList(object, true);
            logger.debug("Putting: {}", keys);
        }
        long started = System.currentTimeMillis();
        ApiServiceTransactionOptions options = new ApiServiceTransactionOptions(ETransactionType.TT_TABLET)
                .setSticky(true);
        if (!atomic) {
            options.setAtomicity(EAtomicity.A_NONE);
        }
        CompletableFuture<Void> r = client.startTransaction(options).thenCompose(t -> {
            ModifyRowsRequest modifyRowsRequest = new ModifyRowsRequest(basePath + "/" + tableMapping.getTableName(),
                    tableMapping.getSchema());
            modifyRowsRequest.addInsert(values);
            modifyRowsRequest.setRequireSyncReplica(false);
            return t.modifyRows(modifyRowsRequest).thenCompose(ignored_void -> t.commit());
        });
        List<Object> finalKeys = keys;
        r.whenComplete((v, t) -> {
            if (t == null) {
                long took = System.currentTimeMillis() - started;
                logger.debug("Put {} in {}", finalKeys, took);
            } else {
                logger.error("Exception on putInternal", t);
            }
        });
        return r;
    }

    private CompletableFuture<Void> putManyInternal(List<T> valueList, YtClient client) {
        if (logger.isDebugEnabled()) {
            List<List<Object>> keys = valueList.stream().map(val -> tableMapping.toValueList(val, true))
                    .collect(Collectors.toList());
            logger.debug("Putting many: {}", keys);
        }
        long started = System.currentTimeMillis();
        ApiServiceTransactionOptions options = new ApiServiceTransactionOptions(ETransactionType.TT_TABLET)
                .setSticky(true);
        if (!atomic) {
            options.setAtomicity(EAtomicity.A_NONE);
        }
        CompletableFuture<Void> r = client.startTransaction(options).thenCompose(t -> {
            ModifyRowsRequest modifyRowsRequest = new ModifyRowsRequest(basePath + "/" + tableMapping.getTableName(),
                    tableMapping.getSchema());
            for (T obj : valueList) {
                List<Object> values = tableMapping.toValueList(obj);
                modifyRowsRequest.addInsert(values);
            }
            modifyRowsRequest.setRequireSyncReplica(false);
            return t.modifyRows(modifyRowsRequest).thenCompose(ignored_void -> t.commit());
        });
        r.whenComplete((v, t) -> {
            if (t == null) {
                long took = System.currentTimeMillis() - started;
                logger.debug("Put many in {}", took);
            } else {
                logger.error("Exception on putManyInternal", t);
            }
        });
        return r;
    }

    private CompletableFuture<Void> deleteInternal(List<Object> keys, YtClient client) {
        logger.debug("Deleting {}", keys);
        long started = System.currentTimeMillis();
        ApiServiceTransactionOptions options = new ApiServiceTransactionOptions(ETransactionType.TT_TABLET)
                .setSticky(true);
        if (!atomic) {
            options.setAtomicity(EAtomicity.A_NONE);
        }
        CompletableFuture<Void> r = client.startTransaction(options).thenCompose(t -> {
            ModifyRowsRequest modifyRowsRequest = new ModifyRowsRequest(basePath + "/" + tableMapping.getTableName(),
                    tableMapping.getSchema());
            modifyRowsRequest.addDelete(keys);
            modifyRowsRequest.setRequireSyncReplica(false);
            return t.modifyRows(modifyRowsRequest).thenCompose(ignored_void -> t.commit());
        });
        r.whenComplete((v, t) -> {
            if (t == null) {
                long took = System.currentTimeMillis() - started;
                logger.debug("Deleted in {}", took);
            } else {
                logger.error("Exception on deleteInternal", t);
            }
        });
        return r;
    }


    private CompletableFuture<YtClient> getReadClient() {
        logger.debug("Getting read client");
        YtClient masterClient = getMasterClient();
        return getReplicationInfo(masterClient).thenApply(ri -> {
            if (!ri.getReplicated() || ri.isAllowsSyncronousReads()) {
                return masterClient;
            } else {
                return factory.getYtClient(ri.getReadClusterNames());
            }
        }).thenCompose(readClient -> ensureSchemaExists(readClient).thenApply(ignored -> readClient));
    }

    private CompletableFuture<YtClient> getWriteClient() {
        logger.debug("Getting write client");
        YtClient client = getMasterClient();
        CompletableFuture<Void> schemaCheck = ensureSchemaExists(client);
        CompletableFuture<Void> atomicityCheck = getAtomicity(client);
        return CompletableFuture.allOf(schemaCheck, atomicityCheck).thenApply(ignored -> client);
    }

    private CompletableFuture<ReplicationInfo> getReplicationInfo(YtClient masterClient) {
        if (replicationInfo == null || replicationInfo.isCompletedExceptionally() ||
                (replicationInfo.isDone() && (System.currentTimeMillis() - replicaSyncTimestamp) > replicaSyncInterval.toMillis())) {
            synchronized (replicationInfoSyncRoot) {
                if (replicationInfo == null || replicationInfo.isCompletedExceptionally() ||
                        (replicationInfo.isDone() && replicationInfo.join().getReplicated() &&
                                (System.currentTimeMillis() - replicaSyncTimestamp) > replicaSyncInterval.toMillis())) {
                    replicationInfo = queryReplicationInfo(masterClient, replicationInfo == null);
                    replicationInfo.whenCompleteAsync(this::onReplicationInfoReady);
                }
            }
        }
        return replicationInfo;
    }

    private void onReplicationInfoReady(ReplicationInfo replicationInfo, Throwable throwable) {
        if (throwable != null) {
            throw new RuntimeException("Could not update replication info", throwable);
        }
        logger.info("Replication info for {} on {} updated: syncReplicas: {}, best async clusters: {}",
                tableMapping.getTableName(), cluster, replicationInfo.isAllowsSyncronousReads(),
                replicationInfo.getReadClusterNames());
        synchronized (replicationInfoSyncRoot) {
            replicaSyncTimestamp = System.currentTimeMillis();
        }
    }

    private CompletableFuture<ReplicationInfo> queryReplicationInfo(YtClient masterClient, boolean verifyType) {
        logger.info("Querying replication info for " + getPath());
        return masterClient.getNode(getPath() + "/@type").thenCompose(node -> {
            if (!node.stringValue().equals("replicated_table")) {
                return CompletableFuture.completedFuture(new ReplicationInfo());
            } else {
                return masterClient.getNode(getPath() + "/@replicas").thenApply(this::readReplicationInfoFromNode);
            }
        }).exceptionally(t -> {
            logger.error("Could not update replication info", t);
            throw new RuntimeException("Could not update replication info", t);
        });
    }

    private CompletableFuture<Void> ensureSchemaExists(YtClient client) {
        if (schemaInfo == null || schemaInfo.isCompletedExceptionally()) {
            synchronized (schemaInfoSyncRoot) {
                if (schemaInfo == null || schemaInfo.isCompletedExceptionally()) {
                    logger.info("Querying schema for " + getPath());
                    schemaInfo = client.getNode(
                            getPath() + "/@schema").thenAccept(tableMapping::loadSchema).handle((r, t) -> {
                        if (t != null) {
                            logger.error("Unable to query schema", t);
                            throw new RuntimeException("Unable to query schema", t);
                        } else {
                            logger.info("Schema loaded for " + getPath());
                            return r;
                        }
                    });
                }
            }
        }
        return schemaInfo;
    }

    private CompletableFuture<Void> getAtomicity(YtClient client) {
        if (atomicityInfo == null || atomicityInfo.isCompletedExceptionally()) {
            synchronized (atomicityInfoSyncRoot) {
                if (atomicityInfo == null || atomicityInfo.isCompletedExceptionally()) {
                    logger.debug("Checking atomicity");
                    atomicityInfo = client.getNode(getPath() + "/@atomicity").thenAccept(tn -> {
                        boolean a = tn.stringValue().equals("full");
                        logger.info(getPath() + " atomicity is {}", a);
                        atomic = a;
                    });
                }
            }
        }
        return atomicityInfo;
    }

    private ReplicationInfo readReplicationInfoFromNode(YTreeNode node) {
        List<Map.Entry<String, YTreeNode>> res = node.asMap().entrySet().stream()
                .filter(entry -> entry.getValue().mapNode().getString("state").equals("enabled") &&
                        entry.getValue().mapNode().getString("replica_path").equals(getPath()))
                .sorted(Comparator.comparingInt(entry -> entry.getValue().mapNode().getInt("replication_lag_time")))
                .collect(Collectors.toList());
        boolean allowsSyncronousReads = node.asMap().entrySet().stream().anyMatch(
                entry -> entry.getValue().mapNode().getString("mode").equals("sync"));
        List<String> readClusterNames = new ArrayList<>();
        Map<String, String> replicaMap = new HashMap<>();
        Map<String, Long> replicaLagTimes = new HashMap<>();
        if (res.size() > 0) {
            long bestLag = res.get(0).getValue().mapNode().getLong("replication_lag_time");
            for (Map.Entry<String, YTreeNode> entry : res) {
                if (entry.getValue().mapNode().getLong("replication_lag_time") == bestLag) {
                    readClusterNames.add(entry.getValue().mapNode().getString("cluster_name"));
                }
                replicaLagTimes.put(entry.getValue().mapNode().getString("cluster_name"),
                        entry.getValue().mapNode().getLong("replication_lag_time"));
                replicaMap.put(entry.getKey(), entry.getValue().mapNode().getString("cluster_name"));
            }
        }
        return new ReplicationInfo(allowsSyncronousReads, readClusterNames, replicaMap, replicaLagTimes);
    }
}
