package ru.yandex.travel.yt.mappings;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Function;

import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.travel.yt.exceptions.InvalidMappingException;
import ru.yandex.travel.yt.exceptions.UnmappedClassException;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedValue;


public class TableMapping<T> {

    public static final String TABLET_INDEX_COLUMN_NAME = "$tablet_index";

    private String tableName = null;
    private long ttl = 0;

    private Map<String, Function<T, Object>> getters = new HashMap<>();
    private Map<String, BiConsumer<T, UnversionedValue>> setters = new HashMap<>();
    private Set<String> mappedColumns = new HashSet<>();
    private Map<String, String> fieldToColumnMap = new HashMap<>();
    private Class<T> mappedClass;
    private TableSchema schema;
    private String timestampColumnName;
    private boolean isOrderedTable;

    public TableMapping(Class<T> mappedClass) {
        this(mappedClass, null, null);
    }

    public TableMapping(Class<T> mappedClass, String tableName, Long ttl) {
        this.mappedClass = mappedClass;
        if (!mappedClass.isAnnotationPresent(YtTable.class)) {
            throw new UnmappedClassException(mappedClass.getSimpleName());
        }
        YtTable tableAnn = mappedClass.getAnnotation(YtTable.class);
        if (!tableAnn.tableName().isEmpty()) {
            this.tableName = tableAnn.tableName();
        }
        if (tableName != null && !tableName.isEmpty()) {
            this.tableName = tableName;
        }

        if (this.tableName == null || this.tableName.isEmpty()) {
            throw new InvalidMappingException("Table name is not specified for class '"
                    + mappedClass.getSimpleName() + "'");
        }

        if (ttl != null) {
            this.ttl = ttl;
        }
        else {
            this.ttl = tableAnn.ttl();
        }

        if (!tableAnn.timestampColumn().isEmpty()) {
            timestampColumnName = tableAnn.timestampColumn();
        }

        if (Arrays.stream(mappedClass.getDeclaredConstructors()).noneMatch(c -> c.getParameterCount() == 0)) {
            throw new InvalidMappingException("Class '" + mappedClass.getSimpleName() + "' should define a " +
                    "parameterless constructor");
        }

        for (Field f : mappedClass.getDeclaredFields()) {
            if (f.isAnnotationPresent(YtColumn.class)) {
                YtColumn columnAnn = f.getAnnotation(YtColumn.class);
                String fieldName = f.getName();
                String columnName = columnAnn.columnName().isEmpty() ? fieldName : columnAnn.columnName();
                if (mappedColumns.contains(columnName)) {
                    throw new InvalidMappingException("Column '" + columnName + "' is mapped more then once");
                }
                mappedColumns.add(columnName);
                fieldToColumnMap.put(fieldName, columnName);
                getters.put(fieldName, createGetter(f));
                setters.put(fieldName, createSetter(f));
            }
        }
    }

    public void setTableName(String tableName) {
        this.tableName = tableName;
    }

    public void setTtl(long ttl) {
        this.ttl = ttl;
    }

    public Set<String> getMappedColumns() {
        return mappedColumns;
    }

    public void resetSchema() {
        schema = null;
    }

    public List<Object> toValueList(T obj) {
        return toValueList(obj, false);
    }

    public List<Object> toValueList(T obj, boolean keysOnly) {
        if (schema == null) {
            throw new RuntimeException("Schema is not loaded");
        }
        TableSchema thisSchema = keysOnly ? schema.toKeys() : schema;

        List<Object> res = new ArrayList<>(thisSchema.getColumnsCount());
        Map<String, Object> map = toValueMap(obj);
        for (String columnName : thisSchema.getColumnNames()) {
            if (mappedColumns.contains(columnName)) {
                res.add(map.get(columnName));
            } else {
                if (columnName.equals(timestampColumnName)) {
                    res.add(System.currentTimeMillis());
                } else if (isOrderedTable && columnName.equals(TABLET_INDEX_COLUMN_NAME)) {
                    res.add(0);
                }
            }
        }
        return res;
    }

    public Optional<T> fromValueList(List<UnversionedValue> valueList) {
        if (schema == null) {
            throw new RuntimeException("Schema is not loaded");
        }
        if (valueList.size() != schema.getColumnsCount()) {
            throw new RuntimeException("Unexpected number of values");
        }
        Map<String, UnversionedValue> map = new HashMap<>(valueList.size());
        for (int i = 0; i < valueList.size(); i++) {
            String columnName = schema.getColumnName(i);
            if (Objects.equals(columnName, timestampColumnName) && ttl > 0) {
                long insertionTimeStamp;
                try {
                    insertionTimeStamp = valueList.get(i).longValue();
                } catch (Exception ex) {
                    throw new RuntimeException("Invalid datatype of timestamp column");
                }
                if (System.currentTimeMillis() - insertionTimeStamp > ttl) {
                    return Optional.empty();
                }
            }
            if (mappedColumns.contains(columnName)) {
                map.put(columnName, valueList.get(i));
            }
        }
        return Optional.of(fromValueMap(map));
    }


    public Map<String, Object> toValueMap(T obj) {
        Map<String, Object> res = new HashMap<>(getters.size());
        for (Map.Entry<String, Function<T, Object>> e : getters.entrySet()) {
            String columnName = fieldToColumnMap.get(e.getKey());
            res.put(columnName, e.getValue().apply(obj));
        }
        return res;
    }

    public T fromValueMap(Map<String, UnversionedValue> valueMap) {
        T res;
        try {
            res = mappedClass.newInstance();
        } catch (Exception e) {
            throw new RuntimeException("Unable to create a new instance of " + mappedClass.getSimpleName(), e);
        }
        for (Map.Entry<String, BiConsumer<T, UnversionedValue>> e : setters.entrySet()) {
            String columnName = fieldToColumnMap.get(e.getKey());
            e.getValue().accept(res, valueMap.get(columnName));
        }
        return res;
    }

    public String getTableName() {
        return tableName;
    }

    public long getTtl() {
        return ttl;
    }

    public TableSchema getSchema() {
        return schema;
    }

    public boolean hasSchema() {
        return (schema != null);
    }

    public void loadSchema(YTreeNode schemaNode) {
        TableSchema.Builder bldr = new TableSchema.Builder();
        boolean hasKeys = false;
        boolean hasTimestamp = false;
        for (YTreeNode columnNode : schemaNode.asList()) {
            Map<String, YTreeNode> column = columnNode.asMap();
            String columnName = column.get("name").stringValue();
            ColumnValueType columnType = ColumnValueType.fromName(column.get("type").stringValue());
            if (column.get("required").boolValue() && !mappedColumns.contains(columnName)) {
                throw new RuntimeException("Required column '" + columnName + "' is not mapped");
            }
            if (columnName.equals(timestampColumnName)) {
                hasTimestamp = true;
            }
            if (!mappedColumns.contains(columnName) && !columnName.equals(timestampColumnName)) {
                // column is not required and is not mapped. Skip it from schema unless it is a timestamp column
                continue;
            }
            if (column.containsKey("sort_order")) {
                hasKeys = true;
                bldr.addKey(columnName, columnType);
            } else {
                bldr.addValue(columnName, columnType);
            }
        }
        if (timestampColumnName != null && !hasTimestamp) {
            throw new RuntimeException("Timestamp column '" + timestampColumnName + "' is not found in schema");
        }
        if (!hasKeys) {
            // ordered tables case - last column will be tablet index
            bldr.setUniqueKeys(false);
            bldr.addValue(TABLET_INDEX_COLUMN_NAME, ColumnValueType.INT64);
            isOrderedTable = true;
        } else {
            isOrderedTable = false;
        }
        schema = bldr.build();
    }


    private Function<T, Object> createGetter(Field f) {
        String getterName = "get" + Character.toUpperCase(f.getName().charAt(0)) + f.getName().substring(1);
        try {
            Method m = mappedClass.getMethod(getterName);
            return (T obj) -> {
                try {
                    return m.invoke(obj);
                } catch (Exception ex) {
                    throw new RuntimeException("Unable to get value of field " + f.getName(), ex);
                }
            };
        } catch (NoSuchMethodException ignored) {
            f.setAccessible(true);
            return (T obj) -> {
                try {
                    return f.get(obj);
                } catch (Exception ex) {
                    throw new RuntimeException("Unable to get value of field " + f.getName(), ex);
                }
            };
        }
    }

    private BiConsumer<T, UnversionedValue> createSetter(Field f) {
        String setterName = "set" + Character.toUpperCase(f.getName().charAt(0)) + f.getName().substring(1);
        try {
            Method m = mappedClass.getMethod(setterName, f.getType());
            return (T obj, UnversionedValue val) -> {
                try {
                    m.invoke(obj, castColumnValueToFieldType(val, f));
                } catch (Exception ex) {
                    throw new RuntimeException("Unable to set value of field " + f.getName(), ex);
                }
            };
        } catch (NoSuchMethodException ignored) {
            f.setAccessible(true);
            return (T obj, UnversionedValue val) -> {
                try {
                    f.set(obj, castColumnValueToFieldType(val, f));
                } catch (Exception ex) {
                    throw new RuntimeException("Unable to set value of field " + f.getName(), ex);
                }
            };
        }
    }

    private Object castColumnValueToFieldType(UnversionedValue unversionedValue, Field field) {
       Object val = retrieveValue(unversionedValue, field);
        if (val instanceof Number) {
            Number value = (Number) val;
            if (field.getType() == Byte.class || field.getType() == byte.class) {
                return value.byteValue();
            }
            if (field.getType() == Short.class || field.getType() == short.class) {
                return value.shortValue();
            }
            if (field.getType() == Integer.class || field.getType() == int.class) {
                return value.intValue();
            }
            if (field.getType() == Long.class || field.getType() == long.class) {
                return value.longValue();
            }
            if (field.getType() == Float.class || field.getType() == float.class) {
                return value.floatValue();
            }
            if (field.getType() == Double.class || field.getType() == double.class) {
                return value.doubleValue();
            }
        }
        return val;
    }

    private static Object retrieveValue(UnversionedValue value, Field f) {
        switch (value.getType()) {
            case STRING:
                if (f.getType().isArray() && f.getType().getComponentType() == byte.class) {
                    return value.bytesValue();
                }
                else {
                    return value.stringValue();
                }
            case BOOLEAN:
                return value.booleanValue();
            case UINT64:
            case INT64:
                return value.longValue();
            case DOUBLE:
                return value.doubleValue();
            default:
                return value.getValue();
        }
    }
}
