import com.typesafe.sbt.less.Import.LessKeys
import com.typesafe.sbt.web.Import.Assets
import com.typesafe.sbt.web.SbtWeb
import play._
import sbt.Keys._
import sbt._
import sbtprotobuf.{ProtobufPlugin => PB}

object Tours extends Build {
  def module(moduleName: String, settings: Seq[Setting[_]] = Nil, dir: String = "") =
    Project(id = moduleName.stripPrefix("tours-"), base = file(if (dir.nonEmpty) dir else moduleName))
      .configs(ItSettings.IntTest)
      .settings(name := moduleName)
      .settings(CommonSettings.buildSettings: _*)
      .settings(settings: _*)

  lazy val root = module("tours", dir = ".", settings = RootSettings.settings)
    .aggregate(protoModel, model, core, storage, api, searcher, wizard, admin, testKit, tools, notifier, calendar,
      warmer, indexer, dataServer, wizardIndexing, partnerMock, adminDb, spark, personal)

  lazy val protoModel = module("tours-model-proto", Protobuf.settings)
  lazy val model = module("tours-model", ToursModel.buildSettings).dependsOn(protoModel, testKit % "test")

  lazy val core = module("tours-core", ToursCore.buildSettings).dependsOn(model, testKit % "test")

  lazy val adminDb = module("tours-admin-db", ToursAdminDb.buildSettings).dependsOn(core, testKit % "test")
  lazy val storage = module("tours-storage", ToursStorage.buildSettings).dependsOn(core, testKit % "test")
  lazy val wizardIndexing = module("tours-wizard-indexing", ToursWizardIndexing.buildSettings).dependsOn(core, testKit % "test")

  lazy val api = module("tours-api", ToursApi.buildSettings).dependsOn(core, testKit % "test")
  lazy val searcher = module("tours-searcher", ToursSearcher.buildSettings).dependsOn(storage, testKit % "test").configs(Wsdl.wsdl)
  lazy val wizard = module("tours-wizard", ToursWizard.buildSettings).dependsOn(core, wizardIndexing, testKit % "test")
  lazy val calendar = module("tours-calendar", ToursCalendar.buildSettings).dependsOn(core, storage, testKit % "test")

  lazy val notifier = module("tours-notifier", ToursNotifier.buildSettings).dependsOn(core, storage, testKit % "test")

  lazy val personal = module("tours-personal", ToursPersonal.buildSettings).dependsOn(core, storage, testKit % "test")

  lazy val admin = module("tours-admin", ToursAdmin.buildSettings)
    .enablePlugins(play.PlayScala)
    .enablePlugins(SbtWeb)
    .dependsOn(core, storage, adminDb, testKit % "test")

  lazy val warmer = module("tours-warmer", ToursWarmer.buildSettings).dependsOn(core, storage, testKit % "test")

  lazy val dataServer = module("tours-data-server", ToursDataServer.buildSettings).dependsOn(core, adminDb, wizardIndexing, testKit % "test")
  lazy val indexer = module("tours-indexer", ToursIndexer.buildSettings).dependsOn(core, adminDb, wizardIndexing, testKit % "test")

  lazy val testKit = module("tours-testkit", ToursTestKit.buildSettings)
  lazy val tools = module("tours-tools", ToursTools.buildSettings).dependsOn(core, indexer, warmer, api, searcher, wizard, admin, testKit % "test")
  lazy val partnerMock = module("tours-partner-mock", ToursPartnerMock.buildSettings).dependsOn(core)

  lazy val spark = module("tours-spark", ToursSpark.buildSettings).dependsOn(core)
}

object CommonSettings {
  import Dependencies._

  val buildVersion = "0.0-SNAPSHOT"

  val buildSettings = Defaults.coreDefaultSettings ++ Seq(
    organization := "ru.yandex.tours",
    version      := buildVersion,
    scalaVersion := "2.11.6",
    scalacOptions:= Seq("-deprecation", "-unchecked", "-feature", "-Xfatal-warnings"),
    resolvers    := Seq(Typesafe, YandexProxy, YandexArtifactory, Mandubian, Cloudera),
    exportJars   := true,
    javacOptions ++= Seq("-source", "1.7", "-target", "1.7", "-encoding", "UTF-8"),
    scalacOptions += "-target:jvm-1.7",
    updateOptions := updateOptions.value.withCachedResolution(true),
    dependencyOverrides ++= overrides
  ) ++ Publish.settings

  //TODO autogenerate from Dependencies
  lazy val overrides = Set(
    scala, scalaReflect, scalaXml,
    jBossLogging, json, guava, zookeeper,
    slf4j, julToSlf4j, jclToSlf4j, logback, logbackCore,
    akkaActor, akkaRemote, akkaSlf4j, akkaTestkit,
    commonsIO, netty, jsr305, javassist, xmlApis, fastUtil,
    hamcrest, curatorXDiscovery, curatorRecipes,
    vertisMonitoring, commonAkkaUtils, extDataProvider
  ) ++ jodaTime ++ jackson ++ spring ++ testDependencies

  lazy val Typesafe = "Typesafe releases" at "https://repo.typesafe.com/typesafe/releases/"
  lazy val YandexProxy = "Yandex proxy" at "http://maven.yandex.net/nexus-mirror/content/groups/public/"
  lazy val YandexArtifactory = "Yandex Artifactory" at "http://artifactory.yandex.net/artifactory/public"
  lazy val Mandubian = "mandubian maven bintray" at "http://dl.bintray.com/mandubian/maven"
  lazy val Cloudera = "Cloudera repository" at "http://repository.cloudera.com/cloudera/libs-release-local/"
}

object RootSettings {
  lazy val settings = Seq(
    Allure.task
  )
}

object ToursModel {
  import Dependencies._

  lazy val moduleDependencies = Seq(
    commonsCodec,
    commonsLang,
    slf4j
  ) ++ jodaTime

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ TestSettings.settings
}

object ToursCore {
  import Dependencies._

  lazy val moduleDependencies = Seq(
    typesafeConfig,
    slf4j,
    json,
    playJson,
    commonsIO,
    akkaActor,
    guava,
    akkaSlf4j,
    logback,
    commonAkkaUtils,
    vertisMonitoring,
    sprayUtils,
//    hdrhistogram,
    extDataProvider,
    fastUtil,
    billingMicroCore,
    billingClient,
    protobufKV,
    asyncHttpClient,
    yoctoDB,
    protoJavaFormat,
    subscriptionsMicroCore,
    vertisScheduler,
    curatorRecipes,
    akkaStream,
    playJsonZipper,
    kryo, kryoSerializers, twitterChill,
    breeze, matrixnetClient, ytDao, s3Auth, scala2j8, grpc, hotels2proto
  ) ++ metrics ++ spray ++ jodaTime ++ spring

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ TestSettings.settings ++ Assembly.settings
}

object ToursApi {

  lazy val moduleDependencies = Seq()

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ Upload.settings ++ TestSettings.settings
}

object ToursSearcher {

  import Dependencies._

  lazy val moduleDependencies = Seq(apacheCxfHttp, apacheCxfJaxWs, commonXml)

  private def genSunmarTask =
    Wsdl.generate(
      "sunmar.wsdl",
      "ru.yandex.tours.partners.sunmar",
      "sunmar_bindings.xml",
      "sunmar_async_bindings.xml"
    )

  private def getOktogoTask =
    Wsdl.generate(
      "oktogo.wsdl",
      "ru.yandex.tours.partners.oktogo",
      "oktogo_bindings.xml",
      "oktogo_async_bindings.xml"
    )

  lazy val buildSettings = Wsdl.settings ++ Seq(
    libraryDependencies ++= moduleDependencies,
    sourceGenerators in Compile <+= genSunmarTask,
    sourceGenerators in Compile <+= getOktogoTask
  ) ++ Upload.settings ++ TestSettings.settings ++ Wsdl.sourceRoot("sunmar.wsdl") ++ Wsdl.sourceRoot("oktogo.wsdl")
}

object ToursCalendar {
  lazy val moduleDependencies = Seq()

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ Upload.settings ++ TestSettings.settings
}

object ToursWizard {

  lazy val moduleDependencies = Seq()

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ Upload.settings ++ TestSettings.settings
}

object ToursStorage {

  lazy val moduleDependencies = Seq()

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ TestSettings.settings
}

object ToursAdminDb {
  import Dependencies._

  lazy val moduleDependencies = Seq(
    snappy,
    slick,
    slickHikariCP,
    mySqlConnector
  )

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ TestSettings.settings
}

object ToursWizardIndexing {

  lazy val moduleDependencies = Seq()

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ TestSettings.settings
}

object ToursNotifier {

  lazy val moduleDependencies = Seq()

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ Upload.settings ++ TestSettings.settings
}

object ToursPersonal {

  lazy val moduleDependencies = Seq()

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ Upload.settings ++ TestSettings.settings
}

object ToursAdmin {
  import Dependencies._

  lazy val moduleDependencies = Seq(
    playFilters
  )

  lazy val moduleOverrides = Set(
    httpCore,
    httpClient
  )

  lazy val ignoredOrganizations = Set(
    "org.seleniumhq.selenium",
    "org.fluentlenium",
    "org.specs2"
  )

  lazy val buildSettings = Play.defaultScalaSettings ++ Seq(
    libraryDependencies ++= moduleDependencies,
    dependencyOverrides ++= moduleOverrides,
    LessKeys.compress in Assets := true,
    libraryDependencies := {
      libraryDependencies.value
        .filterNot { dep => ignoredOrganizations.contains(dep.organization) }
        .filterNot { dep => dep.name.startsWith("play-test") }
    }
  ) ++ Upload.settings ++ TestSettings.settings
}

object ToursWarmer {
  import Dependencies._

  lazy val moduleDependencies = Seq(
    playJson
  )

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ Upload.settings ++ TestSettings.settings
}

object ToursDataServer {
  import Dependencies._

  lazy val moduleDependencies = Seq(
    verba2Common,
    commonsCompress,
    commonXml,
    hydraModel
  )

  lazy val moduleOverrides = Set()

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies,
    dependencyOverrides ++= moduleOverrides
  ) ++ Upload.settings ++ TestSettings.settings
}

object ToursIndexer {
  import Dependencies._

  lazy val moduleDependencies = Seq(
    extDataLoader,
    verba2Common,
    commonsCompress,
    commonXml,
    hydraModel,
    libphonenumber
  )

  lazy val moduleOverrides = Set()

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies,
    dependencyOverrides ++= moduleOverrides
  ) ++ Upload.settings ++ TestSettings.settings
}

object ToursTestKit {
  import Dependencies._

  lazy val moduleDependencies = Seq(
    commonAkkaUtils
  ) ++ metrics

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies,
    libraryDependencies ++= testDependencies
  )
}

object ToursTools {
  import Dependencies._

  lazy val moduleDependencies = Seq(
    hadoopCommon,
    hadoopCore,
    hbaseCommon,
    hbaseClient
  )

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  )
}

object ToursPartnerMock {
  lazy val moduleDependencies = Seq()

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ Upload.settings
}

//under development
object ToursSpark {
  import Dependencies._

  lazy val moduleDependencies = Seq(
    hadoopCore, hadoopHdfs, hadoopCommon, hadoopYarn
  ) ++ spark ++ sparkYarn

  lazy val buildSettings = Seq(
    libraryDependencies ++= moduleDependencies
  ) ++ Upload.settings ++ Assembly.settings
}

object Dependencies {
  lazy val scala = "org.scala-lang" % "scala-library" % "2.11.6"
  lazy val scalaReflect = "org.scala-lang" % "scala-reflect" % "2.11.6"
  lazy val scalaXml = "org.scala-lang.modules" %% "scala-xml" % "1.0.2"
  lazy val scala2j8 = "org.scala-lang.modules" % "scala-java8-compat_2.11" % "0.8.0"

  //logging
  lazy val slf4j = "org.slf4j" % "slf4j-api" % "1.7.7"
  lazy val julToSlf4j = "org.slf4j" % "jul-to-slf4j" % "1.7.7"
  lazy val jclToSlf4j = "org.slf4j" % "jcl-over-slf4j" % "1.7.7"
  lazy val logback = "ch.qos.logback" % "logback-classic" % "1.1.2"
  lazy val logbackCore = "ch.qos.logback" % "logback-core" % "1.1.2"
  lazy val jBossLogging = "org.jboss.logging" % "jboss-logging" % "3.2.0.Final"

  //akka
  lazy val akkaActor = "com.typesafe.akka" %% "akka-actor" % "2.4.1"
  lazy val akkaRemote = "com.typesafe.akka" %% "akka-remote" % "2.4.1"
  lazy val akkaSlf4j = "com.typesafe.akka" %% "akka-slf4j" % "2.4.1"
  lazy val akkaStream = "com.typesafe.akka" %% "akka-stream-experimental" % "2.0.1"
  lazy val akkaTestkit = "com.typesafe.akka" %% "akka-testkit" % "2.4.1"
  lazy val akkaStreamTestkit = "com.typesafe.akka" %% "akka-stream-testkit-experimental" % "2.0.1"

  //spray
  lazy val spray = Seq("spray-can", "spray-http", "spray-routing", "spray-httpx", "spray-client", "spray-caching").map("io.spray" %% _ % "1.3.3")
  lazy val sprayTestkit = "io.spray" %% "spray-testkit" % "1.3.3"

  //play
  lazy val play = "com.typesafe.play" %% "play" % "2.3.8"
  lazy val playTest = "com.typesafe.play" %% "play-test" % "2.3.8"
  lazy val playJdbc = PlayImport.jdbc
  lazy val playJson = PlayImport.json
  lazy val playFilters = PlayImport.filters
  lazy val playWs = PlayImport.ws


  //YANDEX stuff
  lazy val commonAkkaUtils = "ru.yandex" %% "common-akka-util" % "0.14.5" excludeAll (excludeLog4j, excludeMetrics)
  lazy val vertisMonitoring = "ru.yandex.common" %% "vertis-monitoring" % "0.2.53" excludeAll excludeMetrics
  lazy val verba2Common = "ru.yandex.verba2" % "verba2-common" % "0.0.13" excludeAll (exclusions: _*)
  lazy val vertisScheduler = "ru.yandex.vertis" %% "vertis-scheduler" % "0.0.32" excludeAll (exclusions: _*)
  lazy val curatorRecipes = "ru.yandex.vertis" %% "curator-recipes" % "0.0.9" excludeAll (exclusions: _*)
  lazy val billingClient = "ru.yandex.vertis.billing" % "billing-client" % "0.1.3" excludeAll(exclusions: _*)
  lazy val billingMicroCore = "ru.yandex.vertis.billing" % "billing-micro-core" % "0.1.29" excludeAll(exclusions: _*)
  lazy val protobufKV = "ru.yandex.vertis" % "protobuf-kv" % "0.1.3" excludeAll(exclusions: _*)
  lazy val hydraModel = "ru.yandex.vertis.hydra" % "hydra-model" % "0.0.1"
  lazy val ytDao = "ru.yandex" % "travel-old-yt-dao" % "3554793"
  lazy val s3Auth = "ru.yandex" % "travel-old-s3-auth" % "3347353"
  lazy val hotels2proto = "ru.yandex" % "travel-hotels-proto2" % "3904178"

  lazy val extDataLoader = "ru.yandex" % "ext-data-loader" % "0.10.17" excludeAll (excludeMongo, excludeOracle, excludeSvnKit, excludeLog4j, excludeSpring, excludeHadoopHdfs, excludeCommonSharding, excludeMetrics)
  lazy val extDataProvider = "ru.yandex" % "ext-data-provider" % "0.10.17" excludeAll (excludeMongo, excludeOracle, excludeSvnKit, excludeLog4j, excludeSpring, excludeMetrics, excludeCommonsLogging)

  lazy val extDataServer = "ru.yandex.extdata" %% "extdata-server" % "0.0.31" excludeAll (excludeLog4j, excludeSpring, excludeMetrics, excludeCommonsLogging)

  lazy val matrixnetClient = "ru.yandex.vertis.matrixnet" % "scala-matrixnet-client" % "0.0.1"

  lazy val subscriptionsMicroCore = "ru.yandex.vertis.subscriptions" % "subscriptions-micro-core" % "0.0.41" excludeAll excludeMetrics
  lazy val sprayUtils = "ru.yandex.vertis" %% "spray-util" % "0.2.5" excludeAll (excludeMetrics, excludeLog4j)

  lazy val yoctoDB = "com.yandex.yoctodb" % "yoctodb-core" % "0.0.16"

  lazy val commonFramework = "ru.yandex" % "common-framework-lite" % "0.0.6" excludeAll(exclusions: _*)
  lazy val allureAdaptorApi = "ru.yandex.qatools.allure" % "allure-java-adaptor-api" % "1.4.19"
  lazy val commonXml = "ru.yandex.common" %% "common-xml" % "0.1.6"

  //mysql
  lazy val slick = "com.typesafe.slick" %% "slick" % "3.1.0"
  lazy val slickHikariCP = "com.typesafe.slick" %% "slick-hikaricp" % "3.1.0" excludeAll excludeMetrics
  lazy val mySqlConnector = "mysql" % "mysql-connector-java" % "5.1.37"

  //commons
  lazy val commonsLang = "commons-lang" % "commons-lang" % "2.6"
  lazy val commonsIO = "commons-io" % "commons-io" % "2.4"
  lazy val commonsCodec = "commons-codec" % "commons-codec" % "1.9"
  lazy val commonsCompress = "org.apache.commons" % "commons-compress" % "1.9" excludeAll(exclusions: _*)

  lazy val protobuf = "com.google.protobuf" % "protobuf-java" % "2.5.0"
  lazy val protoJavaFormat = "com.googlecode.protobuf-java-format" % "protobuf-java-format" % "1.2"
  lazy val typesafeConfig = "com.typesafe" % "config" % "1.2.1"
  lazy val json = "org.json" % "json" % "20140107"
  lazy val playJsonZipper = "com.mandubian" %% "play-json-zipper" % "1.2"
  lazy val jackson = Seq("jackson-databind", "jackson-annotations").map("com.fasterxml.jackson.core" % _ % "2.3.2")

  lazy val metrics = Seq("metrics-core", "metrics-healthchecks", "metrics-graphite", "metrics-jvm", "metrics-servlets")
    .map("io.dropwizard.metrics" % _ % "3.1.2")

  lazy val guava = "com.google.guava" % "guava" % "20.0"
  lazy val jodaTime = Seq("joda-time" % "joda-time" % "2.9.4", "org.joda" % "joda-convert" % "1.8")
  lazy val hdrhistogram = "org.mpierce.metrics.reservoir" % "hdrhistogram-metrics-reservoir" % "1.1.0"
  lazy val fastUtil = "it.unimi.dsi" % "fastutil" % "6.6.1"

  lazy val libphonenumber = "com.googlecode.libphonenumber" % "libphonenumber" % "7.5.1"

  lazy val xmlApis = "xml-apis" % "xml-apis" % "1.4.01"
  lazy val jsr305 = "com.google.code.findbugs" % "jsr305" % "3.0.0"
  lazy val cglib = "cglib" % "cglib" % "2.2"
  lazy val javassist = "org.javassist" % "javassist" % "3.18.2-GA"
  lazy val kryo = "com.esotericsoftware" % "kryo" % "3.0.1"
  lazy val kryoSerializers = "de.javakaffee" % "kryo-serializers" % "0.37"
  lazy val twitterChill = "com.twitter" %% "chill" % "0.7.2" excludeAll excludeOldKryo

  lazy val netty = "io.netty" % "netty" % "3.10.5.Final"
  lazy val httpCore = "org.apache.httpcomponents" % "httpcore" % "4.3"
  lazy val httpClient = "org.apache.httpcomponents" % "httpclient" % "4.3.1"
  lazy val asyncHttpClient = "com.ning" % "async-http-client" % "1.9.33"

  lazy val apacheCxfHttp = "org.apache.cxf" % "cxf-rt-transports-http-hc" % "3.1.0"
  lazy val apacheCxfJaxRs = "org.apache.cxf" % "cxf-rt-frontend-jaxrs" % "3.1.0"
  lazy val apacheCxfJaxWs = "org.apache.cxf" % "cxf-rt-frontend-jaxws" % "3.1.0"

  //zookeeper
  lazy val zookeeper = "org.apache.zookeeper" % "zookeeper" % "3.4.6" excludeAll (exclusions: _*)
  lazy val curatorXDiscovery = "org.apache.curator" % "curator-x-discovery:" % "2.8.0" excludeAll excludeLog4j
  lazy val curatorTest = "org.apache.curator" % "curator-test" % "2.8.0"

  lazy val spring = Seq("spring-context", "spring-core", "spring-beans", "spring-aop" )
    .map("org.springframework" % _ % "4.1.6.RELEASE" excludeAll excludeCommonsLogging)

  lazy val junit = "junit" % "junit" % "4.11"
  lazy val scalaTest = "org.scalatest" %% "scalatest" % "2.2.2"
  lazy val scalaCheck = "org.scalacheck" %% "scalacheck" % "1.11.3"
  lazy val mockito = "org.mockito" % "mockito-core" % "1.9.5"
  lazy val hamcrest = "org.hamcrest" % "hamcrest-core" % "1.3"
  lazy val jsonAssert = "org.skyscreamer" % "jsonassert" % "1.2.3"

  // last version 0.12 requires "shapeless" % "2.1" which conflicts with spray
  lazy val breeze = "org.scalanlp" %% "breeze" % "0.11.2"

  lazy val snappy = "org.xerial.snappy" % "snappy-java" % "1.1.7.2"

  //hadoop + hbase
  lazy val hadoopCore = "org.apache.hadoop" % "hadoop-mapreduce-client-core" % "2.6.0-cdh5.4.7" excludeAll(exclusions: _*)
  lazy val hadoopHdfs = "org.apache.hadoop" % "hadoop-hdfs" % "2.6.0-cdh5.4.7" excludeAll(exclusions: _*)
  lazy val hadoopCommon = "org.apache.hadoop" % "hadoop-common" % "2.6.0-cdh5.4.7" excludeAll(exclusions: _*)
  lazy val hadoopYarn = "org.apache.hadoop" % "hadoop-yarn-client" % "2.6.0-cdh5.4.7" excludeAll(exclusions: _*)
//  lazy val hadoopYarnServer = "org.apache.hadoop" % "hadoop-yarn-server-common" % "2.6.0-cdh5.4.7" excludeAll(exclusions: _*)
  lazy val hbaseCommon = "org.apache.hbase" % "hbase-common" % "1.0.0-cdh5.4.7" excludeAll(exclusions: _*)
  lazy val hbaseClient = "org.apache.hbase" % "hbase-client" % "1.0.0-cdh5.4.7" excludeAll(exclusions: _*)

  lazy val spark = Seq("spark-core", "spark-sql", "spark-hive", "spark-mllib")
    .map("org.apache.spark" %% _ % "1.6.0" excludeAll (excludeHadoop, excludeLog4j, excludeSlf4jOverLog4j, excludeServletApi))
  lazy val sparkYarn = Seq("spark-yarn")
    .map("org.apache.spark" %% _ % "1.6.0" excludeAll (excludeHadoop, excludeLog4j, excludeSlf4jOverLog4j, excludeServletApi))

  // grpc
  lazy val grpc = "io.grpc" % "grpc-all" % "1.12.0"

  def testDependencies: Seq[ModuleID] = Seq(scalaTest, junit, scalaCheck, mockito, sprayTestkit,
    akkaTestkit, akkaStreamTestkit, curatorTest, jsonAssert, allureAdaptorApi)

  lazy val exclusions = Seq(
    excludeLog4j,
    excludeSlf4jOverLog4j,
    ExclusionRule(name = "commons-codec"),
    ExclusionRule(name = "xml-apis"),
    ExclusionRule(name = "commons-io"),
    excludeCommonsLogging,
    ExclusionRule(name = "commons-lang"),
    ExclusionRule(name = "json"),
    ExclusionRule(name = "spring-context"),
    ExclusionRule(name = "spring-core"),
    ExclusionRule(name = "junit"),
    ExclusionRule(name = "spring-beans"),
    ExclusionRule(name = "httpclient"),
    ExclusionRule(name = "jna"),
    ExclusionRule(name = "ojdbc14"),
    ExclusionRule(name = "svnkit"),
    ExclusionRule(name = "commons-collections"),
    excludeMetrics,
    excludeServletApi
  )

  lazy val excludeSlf4jOverLog4j = ExclusionRule(name = "slf4j-log4j12")
  lazy val excludeJetty = ExclusionRule(organization = "org.eclipse.jetty")
  lazy val excludeServletApi = ExclusionRule(organization = "javax.servlet", name = "servlet-api")
  lazy val excludeMongo = ExclusionRule(organization = "ru.yandex", name = "mongo-utils")
  lazy val excludeHadoopUtil = ExclusionRule(organization = "ru.yandex", name = "hadoop-util-common")
  lazy val excludeOracle = ExclusionRule(name = "ojdbc14")
  lazy val excludeSvnKit = ExclusionRule(name = "svnkit")
  lazy val excludeLog4j = ExclusionRule(organization = "log4j")
  lazy val excludeSpring = ExclusionRule(organization = "org.springframework")
  lazy val excludeHadoop = ExclusionRule(organization = "org.apache.hadoop")
  lazy val excludeHadoopHdfs = ExclusionRule(organization = "ru.yandex.vertis", name = "hadoop-hdfs_2.11")
  lazy val excludeCommonsLogging = ExclusionRule(organization = "commons-logging", name = "commons-logging")
  lazy val excludeCommonSharding = ExclusionRule(organization = "ru.yandex.common", name = "common-sharding")
  lazy val excludeMetrics = ExclusionRule(organization = "com.codahale.metrics")
  lazy val excludeIntellijAnnotations = ExclusionRule(organization = "com.intellij", name = "annotations")
  lazy val excludeOldKryo = ExclusionRule(organization = "com.esotericsoftware.kryo", name = "kryo")
}

object Protobuf {
  def settings: Seq[Setting[_]] = PB.protobufSettings ++ Seq(
    PB.protoc in PB.protobufConfig := {
      val protoc25 = file("/usr/local/Cellar/protobuf@2.5/2.5.0/bin/protoc")
      if (protoc25.exists()) protoc25.toString
      else (PB.protoc in PB.protobufConfig).value
    },
    version in PB.protobufConfig := "2.5.0",
    sourceDirectory in PB.protobufConfig :=  baseDirectory.value / "src/main/proto",
    javaSource in PB.protobufConfig := baseDirectory.value / "target" / "generated-sources" / "proto",
    javaSource in Compile := baseDirectory.value / "target" / "generated-sources" / "proto"
  )
}
