import java.util.jar.JarFile

import sbt.Keys._
import sbt._
import sbtassembly.{AssemblyOption, MergeStrategy, PathList}
import sbtdocker._
import sbtdocker.mutable.Dockerfile
import sbtdocker.Plugin._
import sbtdocker.Plugin.DockerKeys._

import scala.collection.mutable

object Upload {
  lazy val defaultHost = "dev45e.vs.os.yandex.net"

  lazy val upload = inputKey[Unit]("Task for uploading code to host")

  lazy val checkDependencies = taskKey[Unit]("Check dependencies for collisions")
  lazy val copyDependencies = taskKey[java.io.File]("Copying dependencies on lib folder")

  lazy val checkDepTask = checkDependencies := {
    val log = streams.value.log
    val map = new mutable.HashMap[String, File]()
    for (dependency <- (dependencyClasspath in Runtime).value if dependency.data.isFile) {
      val file = dependency.data
      val jar = new JarFile(file)
      try {
        val entries = jar.entries
        Iterator.continually(entries.nextElement()).takeWhile(_ => entries.hasMoreElements).foreach { entry =>
          if (!entry.isDirectory && entry.getName.endsWith(".class")) {
            val old = map.put(entry.getName, file)
            if (old.isDefined) {
              log.error(s"$entry found in 2 files: ${old.get} and $file")
            }
          }
        }
      } finally jar.close()
    }
  }

  lazy val copyDepTask = copyDependencies := {
    val log = streams.value.log
    val target = crossTarget.value
    IO.listFiles(target / "lib").foreach(IO.delete)
    for (dependency <- (dependencyClasspath in Runtime).value) {
      val file = dependency.data
      val destFile = target / "lib" / file.name
      if (file.isFile) {
        IO.copyFile(file, destFile, preserveLastModified = true)
        log.info(s"Copied dependency ${file.name}")
      }
    }
    target / "lib"
  }

  lazy val uploadTask = upload := {
    val target = crossTarget.value / "development"
    IO.delete(target)
    IO.copyDirectory(copyDependencies.value, target / "lib", overwrite = true, preserveLastModified = true)
    val otherFiles = baseDirectory.value / "assembly" / "dev"
    if (otherFiles.exists() && otherFiles.isDirectory) {
      IO.copyDirectory(otherFiles, target, overwrite = true, preserveLastModified = true)
      for (file <- IO.listFiles(target) if file.getName.endsWith(".sh")) {
        s"chmod +x ${file.getAbsolutePath}".!
      }
    }
    val args = Def.spaceDelimited("<host>").parsed
    val host = args.headOption.getOrElse(defaultHost)
    val params = args.tail.headOption.getOrElse("")
    val componentId = name.value
    s"rsync -rvcp $params ${target.getAbsolutePath}/ $host:my_servants/$componentId".!
  }

  lazy val buildDockerTask = dockerfile in docker := {
    val binFiles = baseDirectory.value / "assembly" / "docker"
    val dataDirectory = baseDirectory.value / ".." / "tours-data" / "data"
    val dependencies = copyDependencies.value

    new Dockerfile {
      from("dockerfile/java:oracle-java8")
      run("/bin/sh", "-c", "mkdir -p /etc/yandex && echo docker > /etc/yandex/environment.type")
      copy(dataDirectory, "/tours/data")
      workDir("/tours")
      if (binFiles.exists() && binFiles.isDirectory) {
        copy(binFiles, "/tours/bin")
        run("/bin/sh", "-c", "chmod +x /tours/bin/*.sh")
      }
      copy(dependencies, "/tours/lib")
      cmd("bin/start.sh")
    }
  }

  lazy val imageNameSetting = imageName in docker := {
    ImageName(namespace = Some("tours"),
      repository = name.value.stripPrefix("tours-"),
      tag = Some("latest"))
  }

  lazy val settings = dockerSettings ++ Seq(copyDepTask, checkDepTask, uploadTask, buildDockerTask, imageNameSetting)
}

object Publish {
  def settings: Seq[Def.Setting[_]] = Seq(
    publishMavenStyle := true,
    credentials += Credentials(Path(".") / ".credentials"),
    publishArtifact in (Compile, packageDoc) := false,
    publishTo := {
      val maven = "http://artifactory.yandex.net/artifactory/"
      if (isSnapshot.value) Some("snapshots" at maven + "yandex_vertis_snapshots")
      else Some("releases" at maven + "yandex_vertis_releases")
    }
  )
}

object Assembly {
  import sbtassembly.AssemblyKeys._

  val badPrefixes = Seq("akka-", "kryo", "spark-", "jersey-core-", "reflectasm-1.07-", "stax-api-",
  "commons-beanutils-1.7.0", "parquet-hadoop-bundle-", "asm-3.1", "commons-logging-1.1.3", "netty-all-")
  val goodPrefixes = Seq("kryo-serializers")

  def settings: Seq[Def.Setting[_]] = Seq(
    test in assembly := {},
    assemblyOption in assembly := (assemblyOption in assembly).value.copy(includeScala = false),
    assemblyExcludedJars in assembly := {
      (fullClasspath in assembly).value
        .filter(f => badPrefixes.exists(f.data.getName.startsWith) && !goodPrefixes.exists(f.data.getName.startsWith))
    },
    assemblyMergeStrategy in assembly := {
      case "META-INF/io.netty.versions.properties" ⇒ MergeStrategy.first
      case "apache/hadoop/yarn/util/package-info.class" ⇒ MergeStrategy.first
      case "org/apache/hadoop/yarn/factories/package-info.class" ⇒ MergeStrategy.first
      case "org/apache/hadoop/yarn/factory/providers/package-info.class" ⇒ MergeStrategy.first
      case "org/apache/hadoop/yarn/util/package-info.class" ⇒ MergeStrategy.first
      case "overview.html" ⇒ MergeStrategy.discard
      case PathList("org", "apache", "commons", "collections", _*) ⇒ MergeStrategy.first
      case "plugin.xml" ⇒ MergeStrategy.first
      case x =>
        val oldStrategy = (assemblyMergeStrategy in assembly).value
        oldStrategy(x)
    }
  )
}