import sbt.Keys._
import sbt._

object Wsdl {
  val wsdl = config("wsdl")

  val outDirectory = SettingKey[File]("out-directory")

  private val bootstrapLibs = Set(
    "jaxws-api", "jaxb-api", "saaj-api", "jsr181-api", "javax.annotation", "javax.annotation-api", "webservices-api"
  )

  private def isBootstrapLib(file: File) = bootstrapLibs.exists(file.name.contains)

  lazy val settings = inConfig(wsdl)(Seq(
    outDirectory := baseDirectory.value / "target" / "generated-sources" / "wsdl",
    managedClasspath <<= (classpathTypes, update) map { (ct, report) =>
      Classpaths.managedJars(wsdl, ct, report)
    }
  )) ++ Seq(
    libraryDependencies += "com.sun.xml.ws" % "jaxws-tools" % "2.2.7" % wsdl.name,
    libraryDependencies += "org.jvnet.jaxb2_commons" % "jaxb2-commons-lang" % "2.4" % wsdl.name,
    libraryDependencies += "org.jvnet.jaxb2_commons" % "jaxb2-fluent-api" % "3.0" % wsdl.name,
    libraryDependencies += "org.apache.commons" % "commons-lang3" % "3.2.1"
  )

  def generate(wsdlFileName: String, `package`: String, bindings: String*) = Def.task {
    val resources = (resourceDirectory in Compile).value
    val outDir = (outDirectory in wsdl).value / wsdlFileName
    outDir.mkdirs()

    val tools = {
      val tools0 = file(sys.props("java.home")) / "../lib/tools.jar"
      if (tools0.exists) tools0 else file(sys.props("java.home")) / "lib/tools.jar"
    }
    val invokerCp = classOf[Invoker].getProtectionDomain.getCodeSource.getLocation.toExternalForm
    val cp = (managedClasspath in wsdl).value.map(_.data) :+ tools :+ file(invokerCp)
    val bootstrapCp = cp.filter(isBootstrapLib).map(_.getAbsolutePath).mkString(java.io.File.pathSeparator)
    val classpath = cp.filterNot(isBootstrapLib).map(_.getAbsolutePath).mkString(java.io.File.pathSeparator)

    var args = Seq(
      "-classpath", classpath,
      "-d", outDir.getAbsolutePath,
      "-wsdllocation", "/" + wsdlFileName,
      "-keep",
      "-Xnocompile",
      "-extension",
      "-target", "2.1",
      "-p", `package`,
      "-B-verbose",
      "-B-extension",
      "-B-Xcommons-lang",
      "-B-Xfluent-api"
    )

    bindings.foreach { b =>
      args :+= "-b"
      args :+= (resources / b).getAbsolutePath
    }
    args :+= (resources / wsdlFileName).getAbsolutePath

    val jvmArgs = Seq(
      "-Xbootclasspath/p:" + bootstrapCp,
      "-Djavax.xml.accessExternalSchema=all",
      "-classpath", classpath,
      "Invoker", "com.sun.tools.ws.wscompile.WsimportTool"
    ) ++ args

    Fork.java(
      ForkOptions(),
      jvmArgs
    )

    (outDir ** "*.java").get
  }

  def sourceRoot(wsdlFileName: String) = inConfig(wsdl)(Seq(
    (sourceDirectories in Compile) += (outDirectory in wsdl).value / wsdlFileName,
    (managedSourceDirectories in Compile) += (outDirectory in wsdl).value / wsdlFileName
  ))

}