package ru.yandex.tours.db

import java.io.Closeable

import ru.yandex.tours.db.tables.Clusterization.log
import ru.yandex.tours.util.Logging
import slick.backend.DatabasePublisher
import slick.dbio.{DBIOAction, NoStream, Streaming}
import slick.driver.MySQLDriver.api._
import slick.jdbc.meta.MTable

import scala.concurrent.Future

/**
 * Wrapper for slick's Database to take benefits from mysql's ReplicationDriver
 * https://dev.mysql.com/doc/connector-j/en/connector-j-master-slave-replication-connection.html
 *
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 25.11.15
 */
class DBWrapper(db: Database) extends Closeable with Logging {

  def isReadOnly[R, E <: Effect](a: DBIOAction[R, NoStream, E])(implicit target: Target[E]): Boolean = {
    target.isReadOnly
  }

  private def setReadOnly[E <: Effect, R](a: DBIOAction[R, NoStream, E])(implicit target: Target[E]) = {
    SimpleDBIO[Unit](_.connection.setReadOnly(isReadOnly(a)))
  }

  def run[R, E <: Effect](action: DBIOAction[R, NoStream, E], attempts: Int = 10)(implicit target: Target[E]): Future[R] = {
    db.run {
      setReadOnly(action) >> action
    }.recoverWith({
      case exc =>
        if (attempts > 1) {
          log.info(s"Retry DB query, $attempts attempts left, reason: $exc")
          run(action, attempts - 1)
        } else {
          log.info(s"DB query failed: $exc")
          Future.failed(exc)
        }
    })(db.ioExecutionContext)
  }

  def stream[T, E <: Effect](action: DBIOAction[_, Streaming[T], E])
                            (implicit target: Target[E]): DatabasePublisher[T] = {
    db.stream {
      setReadOnly(action) >> action.withStatementParameters(fetchSize = Integer.MIN_VALUE)
    }
  }

  def createIfNotExists[E <: Table[_]](table: TableQuery[E]): Future[Unit] = {
    val name = table.baseTableRow.tableName
    val q = MTable.getTables(name).flatMap { tables =>
      if (tables.isEmpty) {
        log.info(s"Creating table $name: ${table.schema.createStatements.mkString(";")}")
        table.schema.create
      } else {
        log.info(s"Table $name already exists")
        DBIO.successful(())
      }
    }(db.ioExecutionContext)
    run(q)
  }

  def close(): Unit = db.close()
}

case class Target[E <: Effect](isReadOnly: Boolean)

trait LowPriority {
  implicit def slave[E <: Effect]: Target[E] = Target(true)
}

object Target extends LowPriority {
  implicit def master[E <: Effect.Write]: Target[E] = Target(false)
  implicit def master4schema[E <: Effect.Schema]: Target[E] = Target(false)
  implicit def master4unknownEffect[E <: Effect](implicit ev: =:= [E, Effect]): Target[E] = Target(false)
}