package ru.yandex.tours.db

import ru.yandex.tours.geo
import ru.yandex.tours.model.BaseModel.Point

import scala.collection.immutable.IndexedSeq

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 26.05.16
 */
class GridPoint(val id: Long) extends AnyVal {
  def lonIdx: Int = (id >> 32).toInt
  def latIdx: Int = (id & ((1.toLong << 32) - 1)).toInt

  def nearPoints: IndexedSeq[GridPoint] = (for {
    lo ← lonIdx - 1 to lonIdx + 1
    la ← latIdx - 1 to latIdx + 1
    if la >= 0 && la <= GridPoint.LAT_SIZE
  } yield GridPoint(lo % GridPoint.LON_SIZE, la)).distinct

  override def toString: String = {
    val (lon, lat) = GridPoint.fromIndex(lonIdx, latIdx)
    val maxLon = lon + 1d / GridPoint.MULTIPLIER
    val maxLat = lat + 1d / GridPoint.MULTIPLIER
    s"GridPoint[min = [$lat, $lon], max = [$maxLat, $maxLon], diag = ${geo.distanceInKm(lat, lon, maxLat, maxLon)} km]"
  }
}

object GridPoint {

  private val MIN_LATITUDE = -90
  private val MAX_LATITUDE = 90
  private val MIN_LONGITUDE = -180
  private val MAX_LONGITUDE = 180
  private val MULTIPLIER = 20
  private val LON_SIZE = (MAX_LONGITUDE - MIN_LONGITUDE) * MULTIPLIER
  private val LAT_SIZE = (MAX_LATITUDE - MIN_LATITUDE) * MULTIPLIER

  def apply(lonIdx: Int, latIdx: Int): GridPoint = new GridPoint((lonIdx.toLong << 32) + latIdx)

  private def getIndex(point: Point): (Int, Int) = {
    getIndex(point.getLongitude, point.getLatitude)
  }

  private def getIndex(lon: Double, lat: Double): (Int, Int) = {
    if (lat < MIN_LATITUDE || lat > MAX_LATITUDE) {
      sys.error("wrong latitude value:" + lat)
    }
    if (lon < MIN_LONGITUDE || lon > MAX_LONGITUDE) {
      sys.error("wrong longitude value: " + lon)
    }
    ((lon - MIN_LONGITUDE) * MULTIPLIER).toInt → ((lat - MIN_LATITUDE) * MULTIPLIER).toInt
  }

  def fromPoint(point: Point): GridPoint = {
    val (lonIdx, latIdx) = getIndex(point)
    GridPoint(lonIdx, latIdx)
  }

  def fromPoint(lon: Double, lat: Double): GridPoint = {
    val (lonIdx, latIdx) = getIndex(lon, lat)
    GridPoint(lonIdx, latIdx)
  }

  private def fromIndex(lonIdx: Int, latIdx: Int): (Double, Double) = {
    (lonIdx.toDouble / MULTIPLIER + MIN_LONGITUDE) → (latIdx.toDouble / MULTIPLIER + MIN_LATITUDE)
  }
}