package ru.yandex.tours.db.geomapping

import ru.yandex.tours.geo.mapping.GeoMappingShort
import ru.yandex.tours.model.hotels.Partners.Partner

/**
  * Created by asoboll on 10.12.15.
  */
object GeoMappingRecordProcessor {
  def getMapping(seq: Seq[GeoMappingRecordExt], banned: Set[Int] = Set()) =
    seq.filterNot(banned contains _.geoMappingRecord.geoId).
      groupBy(_.geoMappingRecord.geoId).values.flatMap(applyCommands).map(_.toGeoMapping)

  def getBanned(seq: Seq[(GeoMappingRecord, Long, Boolean)]): Iterable[Int] =
    seq.map(GeoMappingRecordExt.tupled).groupBy(_.geoMappingRecord.geoId).values.flatMap(bannedGeoId)

  def combineDuplicates(list: Iterable[(Partner, String)]): Map[Partner, String] =
    list.groupBy(_._1).mapValues(_.map(_._2).mkString(","))

  private def bannedGeoId(seq: Seq[GeoMappingRecordExt]): Option[Int] = {
    val banned = seq.sorted.reverse.map(_.geoMappingRecord.command).
      find(Seq(Banned, UnBanned) contains _).contains(Banned)
    if (banned) seq.lastOption.map(_.geoMappingRecord.geoId)
    else None
  }

  private def applyDrop(seq: Seq[GeoMappingRecordExt]): Seq[GeoMappingRecordExt] = {
    for {
      mapping <- seq.groupBy(_.fullPartnerId).values.toSeq
      sortedMapping = mapping.filter(_.isManual).sorted.reverse ++
                      mapping.filterNot(_.isManual).sorted.reverse
      record <- sortedMapping.takeWhile(_.geoMappingRecord.command != Drop).lastOption
    } yield record
  }

  private def applyCommands(seq: Seq[GeoMappingRecordExt]): Seq[GeoMappingRecordExt] = {
    seq.filterNot(Seq(Banned, UnBanned) contains _.geoMappingRecord.command).
      groupBy(_.geoMappingRecord.partner).values.toSeq.flatMap(applyDrop)
  }

  private def toSet(mapping: Map[Partner, String]): Set[(Partner, String)] = {
    for {
      (partner, ids) <- mapping.mapValues(_.split(",")).toSet
      id <- ids
    } yield (partner, id)
  }

  def diff(geoId: Int, mapNew: Map[Partner, String], mapOld: Map[Partner, String]): Set[(Command, GeoMappingShort)] = {
    val addSet = for {
      (partner, pId) <- toSet(mapNew) -- toSet(mapOld)
    } yield (Add, GeoMappingShort(partner, geoId, pId))
    val dropSet = for {
      (partner, pId) <- toSet(mapOld) -- toSet(mapNew)
    } yield (Drop, GeoMappingShort(partner, geoId, pId))
    addSet ++ dropSet
  }

  def fullDiff(newData: Iterable[GeoMappingShort],
               oldData: Iterable[GeoMappingShort]): Set[(Command, GeoMappingShort)] = {
    def toMapping(data: Iterable[GeoMappingShort]) =
      data.groupBy(_.geoId).mapValues(_.map { case GeoMappingShort(partner, geoId, pId) => partner -> pId }.toMap)
    val mappingNew = toMapping(newData)
    val mappingOld = toMapping(oldData)
    val geoIdSet = mappingNew.keySet ++ mappingOld.keySet
    for {
      yaId <- geoIdSet
      mapNew = mappingNew.getOrElse(yaId, Map())
      mapOld = mappingOld.getOrElse(yaId, Map())
      diffSet = diff(yaId, mapNew, mapOld)
      record <- diffSet
    } yield record
  }
}
