package ru.yandex.tours.db.geomapping

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.db.{DBWrapper, Transaction, Transactions}
import ru.yandex.tours.geo.mapping.GeoMappingShort
import ru.yandex.tours.model.hotels.Partners.Partner
import slick.driver.MySQLDriver.api._



object GeoMappingTables {
  def getQuery(dataType: DataType) = {
    TableQuery(tag => new GeoMappingTable(dataType)(tag))
  }

  def put(db: DBWrapper, dataType: DataType, transaction: Transaction, data: Iterable[(Command, GeoMappingShort)]) = {
    val query = getQuery(dataType) ++= data.map {
      case (command, geoMappingShort) => GeoMappingRecord(command, geoMappingShort, transaction.id)
    }
    db.run(query)
  }

  private def queryWithCondition(dataType: DataType, condition: GeoMappingTable => Rep[Boolean]) = {
    for {
      geoMapping <- getQuery(dataType)
      transaction <- Transactions.table
      if transaction.isEnabled
      if geoMapping.transactionId === transaction.id
      if condition(geoMapping)
    } yield (geoMapping, transaction.timestamp, transaction.isManual)
  }

  def queryPartner(dataType: DataType, partner: Partner) =
    queryWithCondition(dataType, _.partner === partner.id)

  def queryPartners(dataType: DataType, partners: Iterable[Partner]) =
    queryWithCondition(dataType, _.partner.inSet(partners.map(_.id)))

  def queryByGeoId(dataType: DataType, geoId: Int) =
    queryWithCondition(dataType, _.geoId === geoId)

  def queryPartnerId(dataType: DataType, partner: Partner, geoId: Int) =
    queryWithCondition(dataType, geoMapping =>
      (geoMapping.partner === partner.id) && (geoMapping.geoId === geoId))

  def queryBanned(dataType: DataType) =
    queryWithCondition(dataType, geoMapping =>
      (geoMapping.command === Banned.code) || (geoMapping.command === UnBanned.code))

  def queryBanned(dataType: DataType, ids: List[Int]) = {
    def contains(ids: List[Int], tableId: Rep[Int]): Rep[Boolean] = {
      ids match {
        case List() => tableId =!= tableId
        case id :: idsTail =>
          tableId === id || contains(idsTail, tableId)
      }
    }
    queryWithCondition(dataType, geoMapping =>
      ((geoMapping.command === Banned.code) || (geoMapping.command === UnBanned.code)) && contains(ids, geoMapping.geoId))
  }

  def queryAll(dataType: DataType) = queryWithCondition(dataType, _ => true)

  def queryTransactionIds(dataType: DataType, automaticOnly: Boolean = true, enabledOnly: Boolean = true) = {
    val query = for {
      geoMapping <- getQuery(dataType)
      transaction <- Transactions.table
      if geoMapping.transactionId === transaction.id
      if transaction.isEnabled || !enabledOnly
      if !transaction.isManual || !automaticOnly
    } yield transaction.id
    query.distinct
  }
}