package ru.yandex.tours.db.geomapping

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.db.DBWrapper
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.GeoMappingShort
import ru.yandex.tours.geo.matching.{GeoMatchingHypotheses, HypothesesHolder, Hypothesis}
import ru.yandex.tours.geo.partners.PartnerTrees
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner

import scala.concurrent.{ExecutionContext, Future}
import slick.driver.MySQLDriver.api._

/**
  * Created by asoboll on 12.02.16.
  */
object GeoMappingUtils {
  def DEFAULT_DATATYPES = Iterable(DataTypes.countries, DataTypes.cities, DataTypes.departures)

  def getMappings(db: DBWrapper, dataTypes: Iterable[DataType] = DEFAULT_DATATYPES,
                  partners: Seq[Partner] = Partners.values.toSeq)
                 (implicit ec: ExecutionContext): Future[Map[DataType, Seq[GeoMappingRecordExt]]] = {
    Future.sequence(
      for {
        dataType <- dataTypes
        query = GeoMappingTables.queryPartners(dataType, partners)
      } yield for {
        mapping <- db.run(query.result)
      } yield dataType -> mapping.map(GeoMappingRecordExt.tupled)
    ).map(_.toMap)
  }

  def findDroppedMappings(db: DBWrapper,
                          dataType: DataType,
                          directionPriority: region.Id => Double,
                          trId: Int,
                          partners: Iterable[Partner] = Partners.values)
                         (implicit ec: ExecutionContext): Future[Seq[GeoMappingShort]] = {
    val query = GeoMappingTables.queryPartners(dataType, partners)
    for {
      records <- db.run(query.result)
    } yield {
      val recordsNew = records.map(GeoMappingRecordExt.tupled)
      val recordsOld = recordsNew.filter(_.geoMappingRecord.transactionId <= trId)
      val mappingNew = GeoMappingRecordProcessor.getMapping(recordsNew)
      val mappingOld = GeoMappingRecordProcessor.getMapping(recordsOld)

      val manual = recordsNew.filter { case record: GeoMappingRecordExt =>
        (record.geoMappingRecord.transactionId > trId) && record.isManual
      }.groupBy(_.geoMappingRecord.command)
      val manuallyAdded = manual.getOrElse(Add, Seq.empty)
      val manuallyAddedGeoIds = manuallyAdded.map(r => (r.geoMappingRecord.partner, r.geoMappingRecord.geoId)).toSet
      val manuallyAddedPartnerIds = manuallyAdded.map(_.fullPartnerId).toSet
      val manuallyDropped = manual.getOrElse(Drop, Seq.empty).map(_.toGeoMapping).toSet

      val droppedMapping = for {
        (command, geoMappingShort) <- GeoMappingRecordProcessor.fullDiff(mappingNew, mappingOld)
        if command == Drop
        if !manuallyAddedGeoIds.contains((geoMappingShort.partner, geoMappingShort.geoId))
        if !manuallyAddedPartnerIds.contains((geoMappingShort.partner, geoMappingShort.partnerId))
        if !manuallyDropped.contains(geoMappingShort)
      } yield geoMappingShort
      droppedMapping.toSeq.sortBy(gms => - directionPriority(gms.geoId))
    }
  }

  def buildHypothesis(geoMappingShort: GeoMappingShort,
                      tree: region.Tree,
                      partnerTrees: PartnerTrees,
                      hypotheses: HypothesesHolder = GeoMatchingHypotheses.empty): Option[Hypothesis] = {
    hypotheses.findHypothesis(geoMappingShort).orElse {
      for {
        yandexRegion <- tree.region(geoMappingShort.geoId)
        partnerTree <- partnerTrees.get(geoMappingShort.partner)
        partnerHeader <- partnerTree.header(geoMappingShort.partnerId)
      } yield Hypothesis(yandexRegion, partnerHeader)
    }
  }

  def getTransactionIds(db: DBWrapper,
                        dataTypes: Iterable[DataType] = DEFAULT_DATATYPES,
                        automaticOnly: Boolean = true,
                        enabledOnly: Boolean = true)
                       (implicit ec: ExecutionContext): Future[Seq[Int]] = {
    Future.sequence(
      for {
        dataType <- dataTypes
        query = GeoMappingTables.queryTransactionIds(dataType, automaticOnly, enabledOnly)
      } yield db.run(query.result)
    ).map(_.flatten.toSeq.sorted.distinct)
  }
}