package ru.yandex.tours.db.tables

import java.sql.Timestamp
import java.util.concurrent.atomic.AtomicLong

import ru.yandex.tours.db.tables.HotelAmendments.DbHotelAmending
import ru.yandex.tours.db.{DBWrapper, Transactions}
import ru.yandex.tours.hotels.amendings._
import ru.yandex.tours.model.hotels.HotelsHolder.{ProtoHotelAmending, ProtoHotelAmendments}
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.collections.RafBasedMap
import slick.driver.MySQLDriver.api._

import scala.collection.JavaConverters._
import scala.collection.mutable
import scala.concurrent.{ExecutionContext, Future}

class HotelAmendments(tag: Tag) extends Table[DbHotelAmending](tag, "hotel_amendments") {
  def id = column[Int]("id", O.PrimaryKey, O.AutoInc)

  def hotelId = column[Int]("hotel_id")

  def typeId = column[Int]("type_id")

  def timestamp = column[Timestamp]("timestamp")

  def payload = column[Array[Byte]]("payload")

  def transactionId = column[Int]("transaction_id")

  def hotelIdForeignKey = foreignKey("amendings_hotel_fk", hotelId, Hotels.table)(_.id, ForeignKeyAction.Restrict, ForeignKeyAction.Restrict)

  def transactionForeignKey = foreignKey("amendings_transaction_fk", transactionId, Transactions.table)(_.id, ForeignKeyAction.Restrict, ForeignKeyAction.Restrict)

  override def * = (id, hotelId, typeId, timestamp, payload, transactionId).shaped <>( {
    case (id, hotelId, typeId, timestamp, payload, transactionId) =>
      val amending = HotelAmending.parse(typeId, hotelId, timestamp.getTime, payload)
      DbHotelAmending(id, transactionId, amending)
  }, {
    dbAmending: DbHotelAmending =>
      val amending = dbAmending.amending
      Some(dbAmending.id,
        amending.hotelId,
        amending.`type`.id,
        new Timestamp(amending.timestamp),
        amending.serialize,
        dbAmending.transactionId)
  })
}

object HotelAmendments extends Logging {

  case class DbHotelAmending(id: Int, transactionId: Int, amending: HotelAmending)

  val table = TableQuery[HotelAmendments]

  def getAmendments(db: DBWrapper, hotelIds: Seq[Int])
                   (implicit ec: ExecutionContext): Future[Map[Int, ProtoHotelAmendments]] = {
    val q = table
      .join(Transactions.table).on(_.transactionId === _.id)
      .filter(_._2.isEnabled)
      .map(_._1)
      .filter(_.hotelId.inSet(hotelIds))
      .sortBy(_.hotelId)
      .result
    db.run(q).map(
      rawAmendments => {
        var resultMap = mutable.HashMap.empty[Int, ProtoHotelAmendments]
        var lastHotelId: Option[Int] = None
        val currentAmendments = mutable.Buffer.empty[ProtoHotelAmending]
        def updateAmendments() {
          lastHotelId.foreach { id =>
            assert(currentAmendments.forall(_.getHotelId == id), s"Unexpected amendments for $id: $currentAmendments")
            resultMap.put(id, ProtoHotelAmendments.newBuilder().addAllAmendings(currentAmendments.asJava).build)
          }
        }

        for (row <- rawAmendments) {
          val hotelId = row.amending.hotelId
          if (!lastHotelId.contains(hotelId)) {
            updateAmendments()
            lastHotelId = Some(hotelId)
            currentAmendments.clear()
          }
          currentAmendments += row.amending.toProto
        }
        updateAmendments()
        resultMap.toMap
      }
    )
  }
}
