package ru.yandex.tours.db.tables

import java.sql.Timestamp

import ru.yandex.tours.db.HotelStableIds
import ru.yandex.tours.db.model.DbPartnerHotel
import ru.yandex.tours.model.hotels.HotelsHolder.PartnerHotel
import ru.yandex.tours.util.{GZip, Logging}
import slick.driver.MySQLDriver.api._
import slick.profile.SqlProfile.ColumnOption.SqlType

class Hotels(tag: Tag) extends Table[DbPartnerHotel](tag, "hotels") {
  def id = column[Int]("id", O.PrimaryKey)

  def hotel = column[Array[Byte]]("payload") <>[PartnerHotel, Array[Byte]](Hotels.parseBytes, h => Some(Hotels.toBytes(h)))

  def isNew = column[Boolean]("is_new")

  def isDeleted = column[Boolean]("is_deleted")

  def created = column[Timestamp]("created", SqlType("timestamp not null default CURRENT_TIMESTAMP"))

  def updated = column[Timestamp]("updated", SqlType("timestamp not null default CURRENT_TIMESTAMP on update CURRENT_TIMESTAMP"))

  def idForeignKey = foreignKey("hotel_fk", id, HotelStableIds.table)(_.id, ForeignKeyAction.Restrict, ForeignKeyAction.Restrict)

  override def * = (id, hotel, isNew, isDeleted, created, updated).shaped <> ({
    case (id, hotel, isNew, isDeleted, created, updated) =>
      DbPartnerHotel(id, hotel, isNew, isDeleted, created.getTime, updated.getTime)
  }, {
    dbHotel: DbPartnerHotel =>
      Some(
        dbHotel.id,
        dbHotel.hotel,
        dbHotel.isNew, dbHotel.hotel.getIsDeleted,
        new Timestamp(dbHotel.created),
        new Timestamp(dbHotel.updated)
      )
  })
}

object Hotels extends Logging {

  private def parseBytes(bytes: Array[Byte]): PartnerHotel = PartnerHotel.parseFrom(GZip.decompress(bytes))

  private def toBytes(hotel: PartnerHotel): Array[Byte] = GZip.compress(hotel.toByteArray)

  val table = TableQuery[Hotels]

  def batchUpdate(hotels: Iterable[PartnerHotel]) = {
    SimpleDBIO(jdbc ⇒ {
      val ps = jdbc.connection.prepareStatement(s"update hotels set payload = ?, is_deleted = ? where id = ?")
      for (hotel ← hotels) {
        ps.setBytes(1, toBytes(hotel))
        ps.setBoolean(2, hotel.getIsDeleted)
        ps.setInt(3, hotel.getId)
        ps.addBatch()
      }
      ps.executeBatch()
    })
  }
}