import java.io.File
import java.util.concurrent.Callable

import _root_.filters.{AuthFilter, LoggingFilter}
import com.google.common.cache.CacheBuilder
import controllers.AdminControllers
import play.api
import play.api.Mode._
import play.api.mvc.{EssentialAction, Filters}
import play.api.{Configuration, GlobalSettings, Logger}
import play.filters.gzip.GzipFilter
import ru.yandex.tours.app._
import ru.yandex.tours.db.{DefaultMySqlSupport, HotelsDbSupport}
import ru.yandex.tours.extdata.RemoteExtDataSupport
import ru.yandex.tours.geo.partners.PartnerTreesSupport
import ru.yandex.tours.hotels.enrichers.GeoIdSettersSupport
import ru.yandex.tours.storage.subscriptions.{NotificationStorageSupport, SubscriptionsStorageSupport}
import ru.yandex.tours.util.{Logging, Monitorings}
import ru.yandex.vertis.scheduler.model.{Payload, Schedule, Task, TaskDescriptor}
import sun.misc.Signal

import scala.collection.JavaConversions._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 03.03.15
 */
object Global extends GlobalSettings {

  lazy val application =
    new Application
      with DefaultEnvironment
      with AkkaSupport
      with EmbeddedJettyMonitoring
      with DefaultMetricsSupport
      with DefaultHealthChecksSupport
      with DefaultHttpSupport
      with RemoteExtDataSupport
      with CommonDataHolders
      with PartnerTreesSupport
      with DefaultMySqlSupport
      with DefaultZookeeperSupport
      with SubscriptionsStorageSupport
      with YtSupport
      with NotificationStorageSupport
      with HotelsDbSupport
      with GeoIdSettersSupport
      with RemoteCalendarServiceSupport
      with AdminControllers
      with SchedulerSupport {
      override def lazyLoadResources: Boolean = isEnvironmentLocal

      override def schedulingTasks: Seq[Task] = super.schedulingTasks ++ Seq(
        Task(TaskDescriptor("dummy", Schedule.Manually), Payload.Sync(() => ()))
      )
    }

  lazy val filters = Seq(
    new AuthFilter(application.httpClient, application.config)(application.akkaSystem.dispatcher),
    new LoggingFilter(application.akkaSystem),
    new GzipFilter()
    /*, new CSRFFilter()*/
  )

  Monitorings("admin-monitoring").state("fake").setOk()

  override def onLoadConfig(config: Configuration, path: File, classloader: ClassLoader, mode: Mode): Configuration = {
    new Configuration(application.config)
  }

  override def onStart(app: api.Application): Unit = {
    super.onStart(app)

    Signal.handle(new Signal("HUP"), Logging)

    application.start(interactive = false)

    views.Utils.init(application)

    if (!application.isEnvironmentLocal) {
      application.hotelsIndex
      application.regionTree
      application.geoSuggest
    }

    Logger.info("Application started")
  }

  override def onStop(app: api.Application): Unit = {
    super.onStop(app)

    application.metricsRegistry.getNames
      .foreach(application.metricsRegistry.remove)

    application.stop()

    Logger.info("Application stopped")
  }

  override def doFilter(a: EssentialAction): EssentialAction = {
    Filters(super.doFilter(a), filters: _*)
  }

  private val controllerCache = CacheBuilder.newBuilder().build[Class[_], AnyRef]()

  override def getControllerInstance[A](controllerClass: Class[A]): A = {
    controllerCache.get(controllerClass, new Callable[AnyRef] {
      override def call(): AnyRef = {
        val method = application.getClass.getMethods.find(f => controllerClass.isAssignableFrom(f.getReturnType))
        method match {
          case Some(f) => f.invoke(application)
          case None =>
            Logger.info(s"All fields: ${application.getClass.getFields.mkString("\n")}")
            Logger.info(s"All methods: ${application.getClass.getMethods.mkString("\n")}")
            sys.error(s"Controller of class [$controllerClass] not found")
        }
      }
    }).asInstanceOf[A]
  }
}
