package controllers

import java.io.InputStream

import controllers.suggest.{GeoSuggest, SuggestController}
import ru.yandex.tours.app._
import ru.yandex.tours.backa.BackaPermalinks
import ru.yandex.tours.db.geomapping.DbGeoMappings
import ru.yandex.tours.db.searchsettings.DbSearchSettings
import ru.yandex.tours.db.{HotelsDbSupport, MySqlSupport}
import ru.yandex.tours.extdata.{DataTypes, ExtDataSupport, Utils}
import ru.yandex.tours.geo.base.export.XmlParser
import ru.yandex.tours.geo.base.region.{RegionBoundaries, Tree}
import ru.yandex.tours.geo.matching.GeoMatchingHypotheses
import ru.yandex.tours.geo.partners.PartnerTreesSupport
import ru.yandex.tours.hotels.clustering.ClusteringModel
import ru.yandex.tours.hotels.enrichers._
import ru.yandex.tours.partners.{HotelsCombinedId2Key, PartnerConfigs}
import ru.yandex.tours.storage.subscriptions.{NotificationStorageSupport, SubscriptionsStorageSupport}
import ru.yandex.tours.util.{IO, LabelBuilder}
import ru.yandex.tours.util.naming.HotelsTfIdfModel
import shapeless.HNil

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 22.10.15
 */
trait AdminControllers extends SchedulerTasks {
  this: Application with AkkaSupport
    with CommonDataHolders
    with HttpSupport
    with PartnerTreesSupport
    with ExtDataSupport
    with CalendarServiceSupport
    with MySqlSupport
    with SubscriptionsStorageSupport
    with GeoIdSettersSupport
    with SchedulerSupport
    with HotelsDbSupport
    with NotificationStorageSupport =>

  import akkaSystem.dispatcher

  lazy val labelBuilder = new LabelBuilder(config.getString("tours.label.key"))

  lazy val userService = {
    if (isEnvironmentStable || isEnvironmentTesting) {
      new BlackboxUserService(httpClient, config.getString("auth.blackbox.url"))(dispatcher)
    } else {
      new MockUserService
    }
  }

  lazy val rawGeobaseTree = {
    val dataType = DataTypes.geobase
    val boundaries = RegionBoundaries.from(extDataService, extDataUpdateChecker)
    def parse(is: InputStream): Tree = XmlParser.parse(is, boundaries).get
    Utils.createProxy(
      creator = IO.using(extDataService.readData(dataType))(parse),
      lazyLoad = false,
      dataType,
      dependsOn = Set(RegionBoundaries.dataType),
      onChange = (_: Option[Tree], _: Tree) => (),
      extDataUpdateChecker
    )
  }
  lazy val backaPermalinks = BackaPermalinks.from(extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val tfIdfModel = HotelsTfIdfModel.from(extDataService, extDataUpdateChecker)
  lazy val clusteringModel = ClusteringModel.from(tfIdfModel :: HNil, extDataService, extDataUpdateChecker)

  lazy val subscriptions = new SubscriptionsController(subscriptionStorage, notificationStorage)

  lazy val hotels = new HotelsController(hotelsIndex, db, hotelsDao, partnerTrees, userService,
    clusteringModel, backaPermalinks)

  lazy val calendarController = new CalendarController(calendarService, tourOperators,
    regionTree, geoMappingHolder, directionsStats)

  lazy val labelController = new LabelController(labelBuilder)

  lazy val dbGeoMappings = new DbGeoMappings(db)
  lazy val dbSearchSettings = new DbSearchSettings(db, regionTree)

  lazy val geoMappingController = new GeoMappingController(regionTree, db, dbGeoMappings, dbSearchSettings)
  lazy val regionsController = new RegionsController(regionTree, rawGeobaseTree, db, dbGeoMappings, dbSearchSettings)

  lazy val hypotheses = GeoMatchingHypotheses.from(regionTree :: partnerTrees :: HNil, extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val partnerRegionController = new PartnerRegionsController(regionTree, partnerTrees, hypotheses, db, geoMappingHolder, directionsStats)

  lazy val geoSuggest = GeoSuggest.from(regionTree :: directionsStats :: HNil, extDataUpdateChecker)
  //lazy val partnerGeoSuggestHolder = PartnerGeoSuggestHolder.from(partnerTrees :: HNil, extDataUpdateChecker)
  lazy val suggestController = new SuggestController(geoSuggest, regionTree)

  lazy val partnerSearchController = {
    val hcId2key = HotelsCombinedId2Key.from(extDataService, extDataUpdateChecker)
    new PartnerSearchController(geoMappingHolder, hotelsIndex, regionTree, PartnerConfigs.parse(config), hcId2key)
  }

  lazy val schedulerController = new SchedulerController(taskIsEnabledMap)
}
