package controllers

import play.api.mvc.{Action, Controller}
import ru.yandex.tours.calendar.Calendar.{CalendarEntry, FlightDay}
import ru.yandex.tours.direction.DirectionsStats
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.model.Calendar
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.services.CalendarService
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.collections.SimpleBitSet
import ru.yandex.tours.util.lang.Dates._

import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 13.11.15
 */
class CalendarController(calendar: CalendarService,
                         tourOperators: TourOperators,
                         tree: region.Tree,
                         geoMapping: GeoMappingHolder,
                         directionStats: DirectionsStats)
                        (implicit ec: ExecutionContext) extends Controller {

  //scalastyle:off

  def index() = view(213, 210)

  def view(from: Int, to: Int) = Action.async { implicit req =>
    (tree.region(from), tree.region(to)) match {
      case (Some(fromRegion), Some(toRegion)) =>
        for (entries <- calendar.getHasFlights(from, to)) yield {
          val hasFlights = entries
            .map(documentToCalendarEntry)
            .map(e => e.flightDay.when -> e).toMultiMap
          Ok(views.html.calendar.calendar(fromRegion, toRegion, hasFlights))
        }
      case (None, _) => Future.successful(NotFound(s"Region $from not found"))
      case (_, None) => Future.successful(NotFound(s"Region $to not found"))
    }
  }

  private def documentToCalendarEntry(doc: Calendar.Document) = {
    CalendarEntry(
      FlightDay(doc.getWhen.toLocalDate, doc.getNights),
      SimpleBitSet.from(doc.getHasDeparture).toSet.flatMap(tourOperators.getById),
      SimpleBitSet.from(doc.getNoDeparture).toSet.flatMap(tourOperators.getById)
    )
  }
}
