package controllers

import models._
import play.api.data.Form
import play.api.data.Forms._
import play.api.mvc.{Action, Controller, RequestHeader}
import ru.yandex.tours.backa.BackaPermalinks
import ru.yandex.tours.db._
import ru.yandex.tours.db.dao.HotelsDao
import ru.yandex.tours.db.dao.HotelsDao.WithUrl
import ru.yandex.tours.db.model.DbPartnerHotel
import ru.yandex.tours.db.tables.HotelAmendments.DbHotelAmending
import ru.yandex.tours.db.tables.LinkType.LinkType
import ru.yandex.tours.db.tables._
import ru.yandex.tours.geo.partners.PartnerTrees
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.hotels.amendings._
import ru.yandex.tours.hotels.clustering.{ClusteringContext, ClusteringModel, HotelContext, LocalContext}
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.HotelsHolder.{HotelType, PartnerHotel}
import ru.yandex.tours.model.util.Paging
import ru.yandex.tours.util.collections.RafBasedMap
import ru.yandex.tours.util.parsing.IntValue
import slick.driver.MySQLDriver.api._

import scala.collection.Map
import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Try}
import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 28.10.15
 */
//scalastyle:off
class HotelsController(hotelsIndex: HotelsIndex,
                       dbWrapper: DBWrapper,
                       hotelsDao: HotelsDao,
                       partnerTrees: PartnerTrees,
                       userService: UserService,
                       clusteringModel: ClusteringModel,
                       backaPermalinks: BackaPermalinks)
                      (implicit ec: ExecutionContext) extends Controller {

  import HotelsController._

  def index(filters: HotelFilters, page: Page) = Action { implicit req =>
    val hotels = hotelsIndex.getHotels(page.toPaging, filters.filters: _*)
    val count = hotelsIndex.count(filters.filters: _*)
    Ok(views.html.hotels.hotels_list(hotels, filters, page, count))
  }

  def getHotel(hotelId: Int) = Action { implicit req =>
    hotelsIndex.getHotelById(hotelId) match {
      case None => NotFound(s"Hotel $hotelId not found")
      case Some(hotel) => Ok(views.html.hotels.hotel_page(hotel))
    }
  }

  private def getDbHotel(hotelId: Int) = {
    hotelsDao.get(hotelId)
  }

  def getPartnerHotel(hotelId: Int) = Action.async { implicit req =>
    val result = for {
      hotel <- getDbHotel(hotelId)
      clusters <- Clusterization.retrieveCluster(dbWrapper, hotelId)
    } yield (hotel, clusters)
    result.map {
      case (Some(partnerHotel), clusters) =>
        Ok(views.html.hotels.partner_hotel_page(partnerHotel, clusters))
      case (None, _) => NotFound(s"Hotel $hotelId not found")
    }
  }

  def searchPartnerHotels = Action { implicit req =>
    PartnerHotelRequest.form.bindFromRequest().fold(
      errors => BadRequest("Failed: " + errors),
      req => Redirect(routes.HotelsController.getPartnerHotels(req, Page.default))
    )
  }

  def getPartnerHotels(request: PartnerHotelRequest, page: Page) = Action.async { implicit req =>
    val countFuture = hotelsDao.count(request.toQuery: _*)
    for {
      hotels <- hotelsDao.get(page.toPaging, request.toQuery: _*)
      count <- countFuture
    } yield {
      Ok(views.html.hotels.partner_hotel_list(hotels, count, request, page))
    }
  }

  private def getNear(hotel: DbPartnerHotel): Future[RafBasedMap[Int, PartnerHotel]] = {
    hotelsDao.getNear(hotel.hotel.getRawHotel.getPoint)
  }
  private def getNear2(hotel1: Option[DbPartnerHotel],
                       hotel2: Option[DbPartnerHotel]): Future[Map[Int, PartnerHotel]] = {
    hotel1.filter(h ⇒ HotelsIndex.hasCoord(h.hotel))
      .orElse(hotel2.filter(h ⇒ HotelsIndex.hasCoord(h.hotel)))
      .map(getNear)
      .getOrElse(Future.successful(Map.empty))
  }

  private def closeMap(map: Map[_, _]): Unit = {
    map match {
      case raf: RafBasedMap[_, _] ⇒ raf.close()
      case _ ⇒
    }
  }

  def similarity(hotelId1: Int, hotelId2: Int) = Action.async {
    for {
      result <- getDbHotel(hotelId1) zip getDbHotel(hotelId2)
      nearHotels <- getNear2(result._1, result._2)
    } yield {
      try {
        result match {
          case (Some(hotel1), Some(hotel2)) =>
            val localContext = new LocalContext(nearHotels.valuesIterator)

            val ctx1 = HotelContext(hotel1.hotel, localContext)
            val ctx2 = HotelContext(hotel2.hotel, localContext)

            val clusteringContext = ClusteringContext(ctx1, ctx2)

            val similarity: Double = Try {
              clusteringModel.apply(clusteringContext)
            }.recover {
              case NonFatal(t) =>
                Double.NaN
            }.get

            Ok(similarity.toString)
          case (None, _) => NotFound(s"Hotel $hotelId1 not found")
          case (_, None) => NotFound(s"Hotel $hotelId2 not found")
        }
      } finally {
        closeMap(nearHotels)
      }
    }
  }

  def comparePartnerHotels(hotelId1: Int, hotelId2: Int) = Action.async { implicit req ⇒
    for {
      result <- getDbHotel(hotelId1) zip getDbHotel(hotelId2)
      nearHotels <- getNear2(result._1, result._2)
      links <- Clusterization.retrieveLinks(hotelId1, hotelId2, Clusterization.defaultMinConfidence, dbWrapper).andThen { case Failure(_) ⇒  closeMap(nearHotels) }
      leftCluster <- Clusterization.retrieveCluster(dbWrapper, hotelId1, Clusterization.defaultMinConfidence).andThen { case Failure(_) ⇒  closeMap(nearHotels) }
      link = links.sortBy(l => (l.isManual, l.timestamp)).lastOption
      linkUser <- userService.getUsers(link.toSeq.map(_.author), req.remoteAddress).andThen { case Failure(_) ⇒  closeMap(nearHotels) }
    } yield {
      try {
        result match {
          case (Some(hotel1), Some(hotel2)) =>
            val comparison = HotelComparison(hotel1, hotel2, link, nearHotels, clusteringModel)
            Ok(views.html.hotels.partner_hotel_compare(comparison, link, linkUser.headOption, leftCluster.contains(hotelId2)))
          case (None, _) => NotFound(s"Hotel $hotelId1 not found")
          case (_, None) => NotFound(s"Hotel $hotelId2 not found")
        }
      } finally {
        closeMap(nearHotels)
      }
    }
  }

  def lightComparePartnerHotels(hotelId1: Int, hotelId2: Int) = Action.async { implicit req ⇒
    for {
      result <- getDbHotel(hotelId1) zip getDbHotel(hotelId2)
      links <- Clusterization.retrieveLinks(hotelId1, hotelId2, Clusterization.defaultMinConfidence, dbWrapper)
      leftCluster <- Clusterization.retrieveCluster(dbWrapper, hotelId1, Clusterization.defaultMinConfidence)
      link = links.sortBy(l => (l.isManual, l.timestamp)).lastOption
      linkUser <- userService.getUsers(link.toSeq.map(_.author), req.remoteAddress)
    } yield {
      result match {
        case (Some(hotel1), Some(hotel2)) =>
          val comparison = HotelComparisonLight(hotel1, hotel2, link, clusteringModel)
          Ok(views.html.hotels.partner_hotel_compare_light(comparison, link, linkUser.headOption, leftCluster.contains(hotelId2)))
        case (None, _) => NotFound(s"Hotel $hotelId1 not found")
        case (_, None) => NotFound(s"Hotel $hotelId2 not found")
      }
    }
  }

  def getCluster(hotelId: Int) = Action.async { implicit req =>
    for {
      links <- Clusterization.retrieveClusterLinks(dbWrapper, hotelId, minConfidence = 0d)
      ids = links.flatMap(l => Seq(l.parent, l.child)).toSet + hotelId
      hotels <- hotelsDao.get(ids)
      linkTransactionIds = links.map(_.transactionId).toSet
      linkTransactions <- dbWrapper.run(Transactions.table.filter(_.id.inSet(linkTransactionIds)).result)
      graph = new HotelClusterGraph(links, hotelId, hotels.filter(_.isDeleted).map(_.id).toSet, linkTransactions)
      amendments <- dbWrapper.run(
        HotelAmendments
          .table
          .join(Transactions.table).on(_.transactionId === _.id)
          .filter(_._2.isEnabled)
          .map(_._1)
          .filter(_.hotelId.inSet(graph.coreIds)).result
      )
      amendmentTransactionIds = amendments.map(_.transactionId).toSet
      amendmentTransactions <- dbWrapper.run(Transactions.table.filter(_.id.inSet(amendmentTransactionIds)).result)
      allTransactions = linkTransactions ++ amendmentTransactions
      users <- userService.getUsers(allTransactions.map(_.author).filter(_ > 0), req.remoteAddress)
    } yield {
      if (hotels.size > 100) {
        EntityTooLarge("Too big cluster")
      } else if (hotels.exists(_.id == hotelId)) {
        val cluster = new HotelClusterInfo(hotels, hotelId, graph, amendments, allTransactions, users, backaPermalinks)
        Ok(views.html.hotels.cluster.page(cluster))
      } else {
        NotFound(s"Cluster with hotel $hotelId not found")
      }
    }
  }

  def publish(hotelIds: List[Int]) = Action.async { implicit req =>
    for {
      updated <- hotelsDao.publish(hotelIds)
    } yield {
      val msg = s"Published hotels $hotelIds"
      Ok(msg).flashing("success" -> msg)
    }
  }

  def addHotelFeature(hotelId: Int) = Action.async { implicit request =>
    AddFeatureRequest.form.bindFromRequest().fold(
      errors => Future.successful(BadRequest(errors.toString)),
      req => {
        val values = req.value.split(",").map(_.trim)
        val amending = AddFeatureAmending(hotelId, System.currentTimeMillis(), req.name, values: _*)
        val msg = s"Added feature ${req.name} for cluster $hotelId with values ${values.mkString(", ")}"
        addAmending(hotelId, amending, msg)(request)
      }
    )
  }

  def selectHotelMainImage(hotelId: Int, name: String) = {
    val amending = SetFirstImageAmending(hotelId, System.currentTimeMillis(), name)
    val msg = s"Using image $name for cluster $hotelId"
    addAmending(hotelId, amending, msg)
  }

  def hideHotelImage(hotelId: Int, name: String) = {
    val amending = DeleteImageAmending(hotelId, System.currentTimeMillis(), name)
    val msg = s"Hidden image $name for cluster $hotelId"
    addAmending(hotelId, amending, msg)
  }

  def clearHotelName(hotelId: Int, lang: String) = {
    val amending = DeleteHotelName(hotelId, System.currentTimeMillis(), Languages.withName(lang))
    val msg = s"Cleared $lang name for cluster $hotelId"
    addAmending(hotelId, amending, msg)
  }

  def setHotelName(hotelId: Int) = Action.async { implicit request =>
    setHotelNameForm.bindFromRequest().fold(
      errors => Future.successful(BadRequest(errors.toString)),
      pair => {
        val (lang, value) = pair
        val amending = SetHotelName(hotelId, System.currentTimeMillis(), Languages.withName(lang), value)
        val msg = s"Set $lang name to \042$value\042 for cluster $hotelId"
        addAmending(hotelId, amending, msg)(request)
      }
    )
  }

  def selectHotelFeature(hotelId: Int, name: String, partnerHotelId: Int) = {
    val amending = SelectFeatureAmending(hotelId, System.currentTimeMillis(), name, partnerHotelId)
    val msg = s"Using feature $name for cluster $hotelId from $partnerHotelId"
    addAmending(hotelId, amending, msg)
  }

  def setHotelStars(hotelId: Int) = Action.async { implicit request =>
    setHotelStarsForm.bindFromRequest().fold(
      errors => Future.successful(BadRequest(errors.toString)),
      stars => {
        val amending = SetStarsAmending(hotelId, System.currentTimeMillis(), stars)
        val msg = s"Set stars $stars for cluster $hotelId"
        addAmending(hotelId, amending, msg)(request)
      }
    )
  }

  def selectHotelStars(hotelId: Int, partnerHotelId: Int) = {
    val amending = SelectStarsAmending(hotelId, System.currentTimeMillis(), partnerHotelId)
    val msg = s"Using stars for cluster $hotelId from $partnerHotelId"
    addAmending(hotelId, amending, msg)
  }

  def setHotelType(hotelId: Int) = Action.async { implicit request =>
    setHotelTypeForm.bindFromRequest().fold(
      errors => Future.successful(BadRequest(errors.toString)),
      tpe => {
        val amending = SetHotelTypeAmending(hotelId, System.currentTimeMillis(), tpe)
        val msg = s"Set hotel type to $tpe for cluster $hotelId"
        addAmending(hotelId, amending, msg)(request)
      }
    )
  }

  def selectHotelType(hotelId: Int, partnerHotelId: Int) = {
    val amending = SelectHotelTypeAmending(hotelId, System.currentTimeMillis(), partnerHotelId)
    val msg = s"Using hotel type for cluster $hotelId from $partnerHotelId"
    addAmending(hotelId, amending, msg)
  }

  def selectHotelPoint(hotelId: Int, partnerHotelId: Int) = {
    val amending = SelectPointAmending(hotelId, System.currentTimeMillis(), partnerHotelId)
    val msg = s"Using point for cluster $hotelId from $partnerHotelId"
    addAmending(hotelId, amending, msg)
  }

  def deleteHotelFeature(hotelId: Int, name: String) = {
    val amending = DeleteFeatureAmending(hotelId, System.currentTimeMillis(), name)
    val msg = s"Deleted feature $name from $hotelId"
    addAmending(hotelId, amending, msg)
  }

  def setHotelRegion(hotelId: Int, regionId: Int) = {
    val amending = SetGeoIdAmending(hotelId, System.currentTimeMillis(), regionId)
    val msg = s"Linked $hotelId with region $regionId"
    addAmending(hotelId, amending, msg)
  }

  def addAmending(hotelId: Int, amending: HotelAmending, message: String) = Action.async { implicit req =>
    Transactions.withTransaction(dbWrapper, author = req.uid) { transaction =>
      val dbAmending = DbHotelAmending(-1, transaction.id, amending)

      for {
        updated <- dbWrapper.run(HotelAmendments.table += dbAmending)
      } yield {
        Redirect(routes.HotelsController.getCluster(hotelId)).flashing("success" -> message)
      }
    }
  }

  private def addLink(req: RequestHeader, from: Int, to: Int, linkType: LinkType): Future[Int] = {
    Transactions.withTransaction(dbWrapper, req.uid) { tx =>
      val id1 = from min to
      val id2 = from max to
      val q = Clusterization.table += ClusterLink(-1, id1, id2, tx.id, confidence = 1d, linkType)
      dbWrapper.run(q)
    }
  }

  def linkWith(from: Int, url: String) = {
    url match {
      case AdminHotel(IntValue(id)) => link(from, id)
      case TravelHotel(IntValue(id)) => link(from, id)
      case IntValue(id) => link(from, id)
      case _ =>
        val normalizedUrl = PartnerHotelRequest.unifyPartnerUrl(url)
        Action.async { req =>
          hotelsDao.get(Paging(), WithUrl(normalizedUrl)).flatMap {
            case Seq(hotel) if hotel.id != from =>
              addLink(req, from, hotel.id, LinkType.MERGE).map { _ =>
                val msg = s"Added link $from -> ${hotel.id}"
                Ok(msg).flashing("success" -> msg)
              }
            case Seq(hotel) if hotel.id == from =>
              val msg = s"Cannot link hotel $from with himself"
              Future.successful(BadRequest(msg).flashing("danger" -> msg))
            case _ =>
              val msg = s"Cannot parse hotel id from url: $url"
              Future.successful(BadRequest(msg).flashing("danger" -> msg))
          }
        }
    }
  }

  def link(from: Int, to: Int) = Action.async { req =>
    if (from == to) {
      val msg = s"Cannot link hotel $from with himself"
      Future.successful(BadRequest(msg).flashing("danger" -> msg))
    } else {
      addLink(req, from, to, LinkType.MERGE).map { _ =>
        val msg = s"Added link $from -> $to"
        Ok(msg).flashing("success" -> msg)
      }
    }
  }

  def unlink(from: Int, to: Int) = Action.async { req =>
    addLink(req, from, to, LinkType.UNMERGE).map { _ =>
      val msg = s"Added unlink $from -> $to"
      Ok(msg).flashing("success" -> msg)
    }
  }
}

object HotelsController {
  val AdminHotel = "(?:https?://)?(?:travel-admin|localhost:9000)[^/]*/hotels(?:/cluster|/partner)?/([0-9]+)".r
  val TravelHotel = "(?:https?://)?travel\\.[^/]*/hotel/([0-9]+)(?:[?/].*)?".r

  val setHotelTypeForm = Form(single("type", number.transform[HotelType](HotelType.valueOf, _.getNumber)))
  val setHotelStarsForm = Form(single("stars", number))
  val setHotelNameForm = Form(tuple("lang" → nonEmptyText, "value" → nonEmptyText))
}
