package controllers

import play.api.data._
import play.api.data.Forms._
import play.api.mvc.{Action, Controller}
import ru.yandex.tours.util.{Logging, LabelBuilder}

import scala.util.{Success, Failure, Try}
import spray.http.Uri

class LabelController(labelBuilder: LabelBuilder) extends Controller with Logging {

  val labelForm = Form(
    tuple(
      "method" -> text,
      "label" -> text
    )
  )

  def handleLabel() = Action { implicit request =>
    val (method, label) = labelForm.bindFromRequest.get
    method match {
      case "encode" => Ok(views.html.label(labelBuilder.encrypt(label)))
      case "decode" => Ok(views.html.label(labelOrUrlDecode(label)))
      case _ => BadRequest("Unknown method")
    }
  }

  def viewLabel() = Action { implicit request =>
    Ok(views.html.label(""))
  }

  private def labelOrUrlDecode(label: String) = {
    var l = label
    if (l.startsWith("http://") || l.startsWith("https://")) {
      val uri = Uri(l)
      var p = uri.query.get("label").orElse(uri.query.get("utm_term"))
      if (p.isEmpty) {
        throw new RuntimeException("Cannot find label or utm_term")
      }
      l = p.get
    } else {
      l = l.replace(' ', '+')
        .replace("%2B", "+").replace("%2b", "+")
        .replace("%3D", "=").replace("%3d", "=")
        .replace("%2F", "/").replace("%2f", "/")
    }
    labelBuilder.decrypt(l)
  }

  def decodeLabel(label: String) = Action {
    Try(labelOrUrlDecode(label)) match {
      case Failure(e) => BadRequest("Can not decode label: " + e.getMessage)
      case Success(decoded) => Ok(decoded)
    }
  }
}
