package controllers

import models.{DroppedRegionsRequest, RichRequest}
import play.api.data.Form
import play.api.data.Forms._
import play.api.mvc.{Action, Controller}
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.db.{Transactions, DBWrapper}
import ru.yandex.tours.db.geomapping._
import ru.yandex.tours.direction.DirectionsStats
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.{GeoMappingHolder, GeoMappingShort}
import ru.yandex.tours.geo.matching.{Hypothesis, HypothesesHolder}
import ru.yandex.tours.geo.partners.PartnerTrees
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.parsing.LongValue

import scala.concurrent.{Future, ExecutionContext}
import scala.util.Try

/**
  * Created by asoboll on 16.02.16.
  */
class PartnerRegionsController(tree: region.Tree,
                               partnerTrees: PartnerTrees,
                               hypotheses: HypothesesHolder,
                               db: DBWrapper,
                               geoMappingHolder: GeoMappingHolder,
                               directionsStats: DirectionsStats)
                              (implicit ec: ExecutionContext) extends Controller with Logging {

  def view(partnerName: String, id: String, geoId: Int) = get(partnerName, id, geoId)

  def get(partnerName: String, id: String, geoId: Int = -1) = Action.async { implicit req =>
    Try(Partners.withName(partnerName)).map { partner =>
      for (mappingRecords <- GeoMappingUtils.getMappings(db, partners = Seq(partner))) yield {
        (for {
          partnerTree <- partnerTrees.get(partner)
          partnerRegion <- partnerTree.region(id)
        } yield {
          val mapping = mappingRecords.values.flatMap(GeoMappingRecordProcessor.getMapping(_))
            .map(mappingShort => mappingShort.partnerId -> mappingShort.geoId).toMap
          val mappedRegion = mapping.get(partnerRegion.id).flatMap(tree.region)
          val yandexRegion = tree.region(geoId)
            .orElse(mappedRegion)
            .orElse(hypotheses.byPartnerRegion(partnerRegion).headOption.map(_.yandexRegion))
            .orElse(partnerTree.pathToRoot(partnerRegion).map(_.id).collectFirst(mapping).flatMap(tree.region))
          Ok(views.html.partners.partner_region(partnerRegion, yandexRegion, mappedRegion, mapping, hypotheses))
        }).getOrElse(NotFound("partner region not found"))
      }
    }.getOrElse(Future(NotFound("unknown partner")))
  }

  def saveMapping(partnerName: String, partnerId: String, geoId: Int) = Action.async { implicit req =>
    val saveMappingForm = Form(mapping("command" -> nonEmptyText) { _ match {
        case "add" => Add
        case "drop" => Drop
        case _ => sys.error("Unexpected command")
      }
    }(_ => None))

    val command = saveMappingForm.bindFromRequest.get
    Transactions.withTransaction(db, req.uid) { transaction =>
      val mappingShort = GeoMappingShort(Partners.withName(partnerName), geoId, partnerId)
      val hypothesis = GeoMappingUtils.buildHypothesis(mappingShort, tree, partnerTrees, hypotheses).get
      Future.sequence {
        for (dataType <- GeoMappingUtils.DEFAULT_DATATYPES.filter(hypothesis.hasType))
          yield GeoMappingTables.put(db, dataType, transaction, Iterable((command, mappingShort)))
      }
    }.map { _ =>
      Redirect(routes.PartnerRegionsController.view(partnerName, partnerId, geoId))
        .flashing("success" -> "updated mapping")
    }.recover { case e => InternalServerError("Failed: " + e) }
  }

  def getDroppedMappings(r: DroppedRegionsRequest) = Action.async { implicit req =>
    (for {
      (droppedCountries, nCountries) <- getDroppedHypotheses(DataTypes.countries, r.partners, r.baseTransactionId, r.count)
      (droppedCities, nCities) <- getDroppedHypotheses(DataTypes.cities, r.partners, r.baseTransactionId, r.count)
      transactionIds <- GeoMappingUtils.getTransactionIds(db)
    } yield Ok(views.html.partners.partner_region_dropped(droppedCountries, nCountries, droppedCities, nCities,
                                                          hypotheses, geoMappingHolder, transactionIds))
    ).recover { case e => InternalServerError("Failed: " + e) }
  }

  private def getDroppedHypotheses(dataType: DataType,
                                   partners: Iterable[Partner],
                                   trId: Int,
                                   count: Int): Future[(Seq[Hypothesis], Int)] = {
    def partnersRec = if (partners.isEmpty) Partners.values else partners
    def toHypothesis(geoMappingShort: GeoMappingShort) =
      GeoMappingUtils.buildHypothesis(geoMappingShort, tree, partnerTrees, hypotheses)

    for {
      droppedMapping <- GeoMappingUtils.findDroppedMappings(db, dataType, directionsStats.getPriority, trId, partnersRec)
    } yield (droppedMapping.take(count).flatMap(toHypothesis), droppedMapping.size)
  }
}