package controllers

import java.time.Month

import controllers.PartnerSearchController.PartnerUrl
import models.{LtBash, SearchRequestWrapper}
import org.joda.time.LocalDate
import play.api.mvc.{Action, Controller}
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.search.{ExtendedHotelSearchRequest, ExtendedOfferSearchRequest, SearchType}
import ru.yandex.tours.partners._
import ru.yandex.tours.search.DefaultRequestGenerator
import ru.yandex.tours.util.Logging
import spray.http.Uri

import scala.concurrent.ExecutionContext
import scala.util.Try

//scalastyle:off
class PartnerSearchController(geoMapping: GeoMappingHolder,
                              hotelsIndex: HotelsIndex,
                              tree: Tree,
                              partner2config: Map[Partner, PartnerConfig],
                              hcId2Key: Map[String, String])
                             (implicit ec: ExecutionContext) extends Controller with Logging {

  def searchRequests(wrapper: SearchRequestWrapper) = Action { implicit req =>
    val request = wrapper.searchRequest
    val commonPartners = for {
      (partner, config) <- partner2config
      if Partners.isRoomsPartner(partner)
      if config.useCommonClient
      searchUrl <- config.searchUrl.toIterable
    } yield PartnerUrl(
      partner,
      "search",
      CommonHotelProviderHttp.getSearchUrl(request, searchUrl, partner, hotelsIndex, geoMapping),
      None,
      config.headers
    )

    val hotel = request match {
      case ExtendedOfferSearchRequest(req) => hotelsIndex.getHotelById(req.hotelId)
      case _ => None
    }

    var urls = commonPartners.toVector
    urls +:= PartnerUrl(Partners.booking, "search", BookingHttp.getUri(request, hotelsIndex, geoMapping, tree),
      None, partner2config(Partners.booking).getSearchHeaders)
    urls +:= PartnerUrl(Partners.ostrovokv3, "search", OstrovokV3Http.getSearchUri(request, hotelsIndex, geoMapping),
      None, partner2config(Partners.ostrovokv3).getSearchHeaders)

    val ltHotelId = hotel.flatMap(_.partnerId(Partners.lt).headOption)
    val ltHeaders = partner2config(Partners.lt).getSearchHeaders
    val ltEnqueueUri = LtHttp.createEnqueueUrl(request.hotelRequest.extend(),tree, geoMapping, ltHotelId).map(Uri.apply)
    val ltBash = ltEnqueueUri match {
      case scala.util.Success(uri) => Some(LtBash(uri.toString(), ltHeaders, ltHotelId))
      case scala.util.Failure(_) => None
    }
    urls +:= PartnerUrl(
      Partners.lt,
      "enqueue",
      ltEnqueueUri,
      ltBash,
      ltHeaders
    )
    request match {
      case req: ExtendedHotelSearchRequest =>
        urls +:= PartnerUrl(Partners.hotelsCombined, "multipleSearch",
          HotelsCombinedHttp.multipleHotelRequest(req, hotelsIndex, geoMapping,
            partner2config(Partners.hotelsCombined).searchParameters.get.apiKey.get),
          None,
          HotelsCombinedHttp.authHeaders.drop(1))
      case req: ExtendedOfferSearchRequest =>
        HotelsCombinedHttp.singleHotelRequest(req, hotelsIndex, geoMapping,
          hcId2Key, partner2config(Partners.hotelsCombined).searchParameters.get.apiKey.get).foreach { uri =>
          urls +:= PartnerUrl(Partners.hotelsCombined, "singleSearch", uri, None, HotelsCombinedHttp.authHeaders.drop(1))
        }
      case _ =>
    }

    Ok(views.html.partners.partner_search_request(urls))
  }

}

object PartnerSearchController {
  case class PartnerUrl(partner: Partner, kind: String, url: Try[Uri], bash: Option[String], headers: List[(String, String)] = List.empty)

  def wrapperExample: SearchRequestWrapper =
    SearchRequestWrapper(
      ExtendedHotelSearchRequest(
        DefaultRequestGenerator.getDefaultRequest(
          from = 213,
          to = 239,
          month = Month.of(LocalDate.now.getMonthOfYear),
          context = SearchType.ROOMS
        )))
}
