package controllers

import models.{CustomRegionForm, Page}
import play.api.libs.json.{JsString, Json}
import play.api.mvc.{Action, Controller}
import ru.yandex.tours.db.geomapping.DbGeoMappings
import ru.yandex.tours.db.searchsettings.DbSearchSettings
import ru.yandex.tours.db.{SearchSettings, CustomRegions, DBWrapper}
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.model.search.{SearchOptions, SearchType}
import ru.yandex.tours.util.Logging
import slick.driver.MySQLDriver.api._

import scala.concurrent.{Future, ExecutionContext}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 19.11.15
 */
class RegionsController(tree: region.Tree,
                        rawGeobase: region.Tree,
                        db: DBWrapper,
                        dbGeoMappings: DbGeoMappings,
                        searchSettings: DbSearchSettings)
                       (implicit ec: ExecutionContext) extends Controller with Logging {

  db.createIfNotExists(CustomRegions.table)
  db.createIfNotExists(SearchSettings.table)

  def index() = Action.async { implicit req =>
    val rootRegion = tree.root.get
    for {
      regionMappings <- dbGeoMappings.getRegionMapping(rootRegion.id, tree)
      regionSearchSettings <- searchSettings.getRegionSearchSettings(rootRegion)
    } yield
      Ok(views.html.regions.region(rootRegion, regionMappings, regionSearchSettings))
  }

  def custom(page: Page) = Action.async { implicit req =>
    for {
      count <- db.run(CustomRegions.table.size.result)
      result <- db.run(CustomRegions.table.drop(page.skip).take(page.size).result)
    } yield {
      Ok(views.html.regions.custom_regions(result, page, count))
    }
  }

  def get(id: Int) = Action.async { implicit req =>
    tree.region(id) match {
      case Some(region) =>
        for {
          regionMappings <- dbGeoMappings.getRegionMapping(region.id, tree)
          regionSearchSettings <- searchSettings.getRegionSearchSettings(region)
        } yield
          Ok(views.html.regions.region(region, regionMappings, regionSearchSettings))
      case None =>
        val rootRegion = tree.root.get
        for {
          regionMappings <- dbGeoMappings.getRegionMapping(rootRegion.id, tree)
          regionSearchSettings <- searchSettings.getRegionSearchSettings(rootRegion)
        } yield
          NotFound(views.html.regions.region(rootRegion, regionMappings, regionSearchSettings))
    }
  }

  def createNew = Action { implicit req =>
    Ok(views.html.regions.edit_region(None, None))
  }

  def edit(id: Int) = Action.async { implicit req =>
    for (customRegions <- db.run(CustomRegions.table.filter(_.id === id).result)) yield {
      val region = rawGeobase.region(id)
      if (region.isEmpty && customRegions.isEmpty) {
        Redirect(routes.RegionsController.index())
          .flashing("warning" -> s"Region $id not found")
      } else {
        Ok(views.html.regions.edit_region(region, customRegions.headOption))
      }
    }
  }

  def save = Action.async { implicit req =>
    CustomRegionForm.form.bindFromRequest().fold(
      errors => Future.successful(Ok("Failed: " + errors)),
      region => {
        val q =
          if (region.id <= 0) {
            (CustomRegions.table returning CustomRegions.table.map(_.id)) += region
          } else {
            CustomRegions.table.insertOrUpdate(region).map(_ => region.id)
          }
        for(newId <- db.run(q)) yield {
          Redirect(routes.RegionsController.custom())
            .flashing("success" -> s"Saved region with id $newId")
        }
      }
    )
  }

  def delete(id: Int) = Action.async { implicit req =>
    for (deleted <- db.run(CustomRegions.table.filter(_.id === id).delete)) yield {
      val res = Redirect(routes.RegionsController.custom())

      if (deleted > 0) res.flashing("success" -> s"Deleted region with id $id")
      else res.flashing("warning" -> s"Nothing deleted")
    }
  }

  def putSearchSettings(regionId: Int) = Action.async { implicit req =>
    req.body.asJson.fold {
      Future.successful(BadRequest(Json.obj("error" -> JsString("JSON object is expected"))))
    } {json =>
      (json \ "search_option").asOpt[String].fold {
        searchSettings.deleteSearchSettings(regionId,
          SearchType.withName((json \ "search_type").as[String])).map{ _ =>
          Ok(Json.obj())
        }
      } { searchOption =>
        searchSettings.updateSearchSettings(
          regionId,
          SearchType.withName((json \ "search_type").as[String]),
          SearchOptions.withName(searchOption)
        ).map { _ =>
          Ok(Json.obj())
        }
      }
    }
  }
}
