package controllers

import java.util.concurrent.TimeUnit

import com.google.common.cache.{CacheBuilder, CacheLoader}
import com.google.common.util.concurrent.ListenableFuture
import play.api.mvc.{Action, Controller}
import ru.yandex.tours.model.subscriptions.{Subscription, UserIdentity}
import ru.yandex.tours.storage.subscriptions.{NotificationStorage, SubscriptionStorage}
import ru.yandex.tours.util.lang.Futures._

import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 22.10.15
 */
class SubscriptionsController(subscriptionStorage: SubscriptionStorage, notificationStorage: NotificationStorage)(implicit ec: ExecutionContext) extends Controller {

  private val allSubscriptionsCache = CacheBuilder.newBuilder()
    .refreshAfterWrite(1, TimeUnit.MINUTES)
    .build(new CacheLoader[String, Seq[Subscription]] {
      override def load(key: String): Seq[Subscription] = {
        Await.result(subscriptionStorage.getAllSubscriptions, 1.minute)
      }
      override def reload(key: String, oldValue: Seq[Subscription]): ListenableFuture[Seq[Subscription]] = {
        subscriptionStorage.getAllSubscriptions.asGuavaFuture
      }
    })

  def all = Action { implicit req =>
    val subscriptions = allSubscriptionsCache.get("")
    Ok(views.html.subscriptions.subscriptions_list(subscriptions, Option.empty))
  }

  def forUser(identityType: String, identity: String) = Action.async { implicit req =>
    val userIdentity = UserIdentity(identityType, identity)
    for (subscriptions <- subscriptionStorage.getSubscriptions(userIdentity)) yield {
      Ok(views.html.subscriptions.subscriptions_list(subscriptions, Option(userIdentity)))
    }
  }

  def get(identityType: String, identity: String, id: String) = Action.async { implicit req =>
    val userIdentity = UserIdentity(identityType, identity)
    subscriptionStorage.getSubscription(userIdentity, id).flatMap {
      case Some(subscription) =>
        notificationStorage.getLastNotification(subscription).map { lastNotification =>
          Ok(views.html.subscriptions.subscription(subscription, lastNotification))
        }
      case None =>
        Future.successful {
          NotFound(s"Subscription $userIdentity/$id not found")
        }
    }
  }
}
