package controllers

import akka.util.Timeout
import models.User
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.http.AsyncHttpClient
import spray.http.Uri

import scala.collection.mutable
import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 28.02.16
 */
trait UserService {
  def getUser(uid: Long, userIp: String): Future[Option[User]]
  def getUsers(uids: Seq[Long], userIp: String): Future[Seq[User]]
}

class BlackboxUserService(httpClient: AsyncHttpClient, blackboxUrl: String)
                         (implicit ex: ExecutionContext) extends UserService with Logging {

  private implicit val timeout = Timeout(10.seconds)

  override def getUser(uid: Long, userIp: String): Future[Option[User]] = {
    getUsers(Seq(uid), userIp).map(_.headOption)
  }

  override def getUsers(uids: Seq[Long], userIp: String): Future[Seq[User]] = {
    if (uids.isEmpty) {
      return Future.successful(Seq.empty)
    }

    val distinctUids = uids.distinct
    val uri = Uri(blackboxUrl).withQuery(
      "method" -> "userinfo",
      "userip" -> userIp,
      "uid" -> distinctUids.mkString(",")
    )
    httpClient.get(uri).map { case (_, response) =>
      val xml = scala.xml.XML.loadString(response)
      val users = if (distinctUids.size > 1) xml \\ "user" else xml
      val result = for {
        user <- users
        id <- user \ "uid"
        login <- user \ "login"
      } yield User(id.text.toLong, login.text)

      if (result.isEmpty) {
        log.warn(s"Cannot parse users ($distinctUids) from blackbox answer $response")
      }
      result
    }
  }
}

class MockUserService(implicit ec: ExecutionContext) extends UserService {
  private val logins = new mutable.HashMap[Long, String]()

  override def getUser(uid: Long, userIp: String): Future[Option[User]] = {
    if (uid <= 0) return Future.successful(None)
    val login = logins.getOrElseUpdate(uid, "User " + (logins.size + 1))
    Future.successful(Some(User(uid, login)))
  }

  override def getUsers(uids: Seq[Long], userIp: String): Future[Seq[User]] = {
    Future.traverse(uids)(getUser(_, userIp)).map(_.flatten)
  }
}