package filters

import akka.util.Timeout
import com.typesafe.config.Config
import play._
import play.api.Play._
import play.api.mvc._
import ru.yandex.tours.util.http.AsyncHttpClient
import spray.http.Uri

import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration._
import scala.language.postfixOps

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 23.07.14 19:47
 */
class AuthFilter(httpClient: AsyncHttpClient, config: Config)
                (implicit ec: ExecutionContext) extends Filter {

  private val systemPaths = Set("/ping", "/monrun")
  private val ignoredPaths = Set("/label/decode", "/api/")
  private val adminPaths = Set("/subscriptions", "/partners", "/scheduler", "/label")
  private val protocol = if (config.getBoolean("auth.isHttps")) "https://" else "http://"
  private val passportUrl = config.getString("auth.passport.url")
  private val blackboxUrl = config.getString("auth.blackbox.url")
  private val allowedLogins = parseLogins(config.getString("tours.admin.logins"))
  private val allowedAdminLogins = parseLogins(config.getString("tours.admin.admin_logins"))
  private val timeout = Timeout(10.seconds)

  private def parseLogins(logins: String): Set[String] = {
    if (logins.isEmpty) {
      Set.empty
    } else {
      logins.split(",").map(_.trim).toSet
    }
  }

  private def checkLogin(login: String, possible: Set[String]): Boolean = {
    if (possible.isEmpty) {
      true
    } else {
      possible.contains(login)
    }
  }

  override def apply(next: (RequestHeader) => Future[Result])(req: RequestHeader): Future[Result] = {
    if (systemPaths.exists(p ⇒ req.path.startsWith(p))) {
      next(req)
    } else if (isDev(current) || System.getProperty("skip_auth") == "1") {
      next(withLogin(req, System.getProperty("user.name")))
    } else {
      checkBlackboxSession(req)(next)
    }
  }

  def withLogin(req: RequestHeader, login: String, uid: String = "-1"): RequestHeader = {
    req.copy(tags = req.tags + (Security.username -> login) + ("uid" -> uid))
  }

  // check auth in blackbox and receive user name
  def checkBlackboxSession(request: RequestHeader)(next: (RequestHeader) => Future[Result]): Future[Result] = {
    val sessionCookie = request.cookies.get("Session_id").getOrElse(Cookie("", ""))
    val sessionid2Cookie = request.cookies.get("sessionid2").getOrElse(Cookie("", ""))

    Logger.debug(s"request to blackbox $blackboxUrl; return url=${request.uri}")

    val blackboxRequest =
      httpClient.get(Uri(blackboxUrl).withQuery(
        "mode" -> "sessionid",
        "userip" -> request.remoteAddress,
        "sessionid" -> sessionCookie.value,
        "sslsessionid" -> sessionid2Cookie.value,
        "host" -> request.domain
      ))(timeout)

    blackboxRequest.flatMap { case (_, response) =>
      val xml = scala.xml.XML.loadString(response)
      (xml \\ "login" headOption).map(_.text).map {
        login =>
          var allow = checkLogin(login, allowedLogins)
          val isAdminPath = adminPaths.exists(p ⇒ request.path.startsWith(p))
          if (allow && isAdminPath) {
            allow = checkLogin(login, allowedAdminLogins)
          }
          if (allow) {
            val uid = (xml \\ "uid" head).text
            Logger.debug(s"login: $login; uid=$uid")
            next(withLogin(request, login, uid))
          } else {
            Logger.info(s"Login $login is not allowed")
            Future.successful { Results.Forbidden }
          }
      }.getOrElse {
        if (ignoredPaths.exists(p ⇒ request.path.startsWith(p))) {
          next(request)
        } else {
          Logger.warn(s"some error from blackbox: $response -> go to passport")
          Future.successful {
            Results.Redirect(passportUrl,
              Map("mode" -> Seq("auth"), "retpath" -> Seq(protocol + request.host + request.uri)))
          }
        }
      }
    }
  }
}
