package filters

import akka.actor.ActorSystem
import models.RichRequest
import play.api.mvc.{Filter, RequestHeader, Result}
import play.api.{Logger, Routes}

import scala.concurrent.Future

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 28.08.14 20:47
 */
class LoggingFilter(akkaSystem: ActorSystem) extends Filter {
  import akkaSystem.dispatcher

  private val logger = Logger.apply("http.access.logger")

  override def apply(next: (RequestHeader) => Future[Result])(req: RequestHeader): Future[Result] = {
    val startTime = System.nanoTime()
    def requestTime = (System.nanoTime() - startTime) / 1000000

    val action = req.tags.getOrElse(Routes.ROUTE_CONTROLLER, "") + "." + req.tags.getOrElse(Routes.ROUTE_ACTION_METHOD, "")

    def logSuccess(result: Result) = {
      logger.info(s"${req.remoteAddress}\t${req.method}\t" +
        s"${req.host}${req.uri}\t${req.username}\t$action\t$requestTime\t${result.header.status}")

      result.withHeaders("Request-Time" -> requestTime.toString)
    }
    def logFailure(e: Throwable) = {
      logger.info(s"${req.remoteAddress}\t${req.method}\t" +
        s"${req.uri}\t${req.username}\t$action\t$requestTime\t500")
      e
    }

    next(req).transform(logSuccess, logFailure)
  }
}
