package models

import org.joda.time.DateTimeZone
import play.api.data.Mapping
import ru.yandex.tours.geo.base.custom.PartialRegion
import ru.yandex.tours.geo.base.region.Types
import ru.yandex.tours.model.{Languages, LocalizedString, MapRectangle}
import ru.yandex.tours.util.parsing.IntValue

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 30.11.15
 */
object CustomRegionForm {
  import play.api.data.Form
  import play.api.data.Forms._

  private def doubleValue = bigDecimal.transform[Double](_.doubleValue(), BigDecimal.apply)
  private val tzMapping = nonEmptyText.transform[DateTimeZone](DateTimeZone.forID, _.getID)

  val boundingBoxMapper: Mapping[MapRectangle] = mapping(
    "min_lon" -> doubleValue,
    "min_lat" -> doubleValue,
    "max_lon" -> doubleValue,
    "max_lat" -> doubleValue
  )(MapRectangle.apply)(MapRectangle.unapply)

  val nameMapping = mapping(
    "lang" -> nonEmptyText.transform[Languages.Value](Languages.withName, _.toString),
    "name" -> text
  )(Tuple2.apply)(Tuple2.unapply)

  val localizedStringMapper = list(nameMapping).transform[LocalizedString](
    list => LocalizedString(list.filter(_._2.nonEmpty): _*),
    ls => ls.values.toList
  )

  val partialRegionMapper = mapping(
    "type" -> optional(nonEmptyText.transform[Types.Value](Types.withName, _.toString)),
    "parent_id" -> optional(number),
    "name" -> optional(localizedStringMapper).transform[Option[LocalizedString]](_.filterNot(_.isEmpty), identity),
    "genitive" -> optional(nonEmptyText),
    "dative" -> optional(nonEmptyText),
    "accusative" -> optional(nonEmptyText),
    "locative" -> optional(nonEmptyText),
    "preposition" -> optional(nonEmptyText),
    "synonyms" -> text.transform[Set[String]](_.split(",").filter(_.nonEmpty).map(_.trim).toSet, _.mkString(",")),
    "longitude" -> optional(doubleValue),
    "latitude" -> optional(doubleValue),
    "bounding_box" -> optional(boundingBoxMapper).transform[Option[MapRectangle]](_.filterNot(_.isEmpty), identity),
    "time_zone" -> optional(tzMapping)
  )(PartialRegion.apply)(PartialRegion.unapply)

  val customRegionMapper = mapping(
    "id" -> number(),
    "is_new" -> boolean,
    "custom_region" -> partialRegionMapper,
    "children" -> text.transform[Seq[Int]](_.split(",").filter(_.nonEmpty).map(_.trim).flatMap(IntValue.apply), _.mkString(","))
  )(CustomRegion.apply)(CustomRegion.unapply)

  val form = Form(customRegionMapper)
}
