package models

import play.api.mvc.QueryStringBindable
import ru.yandex.tours.model.hotels.Partners.Partner

import scala.util.{Failure, Success, Try}

/**
  * Created by asoboll on 15.03.16.
  */
case class DroppedRegionsRequest(partners: List[Partner], baseTransactionId: Int, count: Int)

object DroppedRegionsRequest {
  val PARTNERS = Partners.values.toSeq.sorted
  val COUNT_OPTIONS = Seq(20, 100, 500, 2000)
  val DEFAULT_TRANSACTION = 1
  val DEFAULT_COUNT = 100
  val PARTNERS_PARAM = "partner"
  val TRANSACTION_PARAM = "transactionId"
  val COUNT_PARAM = "count"
  def empty = DroppedRegionsRequest(List.empty[Partner], DEFAULT_TRANSACTION, DEFAULT_COUNT)

  private val intBinder = implicitly[QueryStringBindable[Int]]

  implicit object PartnerBinder extends QueryStringBindable[Partner] {
    override def bind(key: String, params: Map[String, Seq[String]]): Option[Either[String, Partner]] =
      intBinder.bind(key, params).map(_.right.map(Partners.apply))

    override def unbind(key: String, value: Partner): String =
      intBinder.unbind(key, value.id)
  }

  implicit object DroppedRegionsRequestBinder extends QueryStringBindable[DroppedRegionsRequest] {
    private def defaultRequest = Some(Right(DroppedRegionsRequest.empty))
    val partnersBinder = implicitly[QueryStringBindable[List[Partner]]]

    override def bind(key: String, params: Map[String, Seq[String]]): Option[Either[String, DroppedRegionsRequest]] = {
      val prefix = if (key.isEmpty) "" else s"$key."
      Try {
        for {
          partners <- partnersBinder.bind(prefix + PARTNERS_PARAM, params).getOrElse(Right(List.empty[Partner])).right
          baseTransactionId <- intBinder.bind(prefix + TRANSACTION_PARAM, params).getOrElse(Right(DEFAULT_TRANSACTION)).right
          count <- intBinder.bind(prefix + COUNT_PARAM, params).getOrElse(Right(DEFAULT_COUNT)).right
        } yield DroppedRegionsRequest(partners, baseTransactionId, count)
      } match {
        case Success(q) => Some(q)
        case Failure(e) => Some(Left(e.getMessage))
      }
    }

    override def unbind(key: String, value: DroppedRegionsRequest): String = {
      if (value == DroppedRegionsRequest.empty) return ""
      val prefix = if (key.isEmpty) "" else s"$key."
      partnersBinder.unbind(prefix + PARTNERS_PARAM, value.partners) + "&" +
      intBinder.unbind(prefix + TRANSACTION_PARAM, value.baseTransactionId) + "&" +
      intBinder.unbind(prefix + COUNT_PARAM, value.count)
    }
  }
}