package models

import ru.yandex.tours.db.tables.{Clusterization, ClusterLink, LinkType}
import ru.yandex.tours.db.Transaction
import ru.yandex.tours.util.collections.Graph

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 28.02.16
 */
class HotelClusterGraph(links: Seq[ClusterLink],
                        startId: Int,
                        deletedSet: Set[Int],
                        transactions: Seq[Transaction]) {

  private val transactionsMap = transactions.map(t => t.id -> t).toMap

  private val sortedLinks =
    links.sortBy { l => val t = transactionsMap(l.transactionId); (t.isManual, t.timestamp) }

  private val lastLinks = sortedLinks.groupBy(l => Seq(l.parent, l.child).sorted)
    .map { case (g, ls) => g -> ls.last }

  val linksWithStatus = sortedLinks.map { link =>
    val isLast = lastLinks(Seq(link.parent, link.child).sorted) == link
    link -> isLast
  }

  val activeLinks = linksWithStatus.filter(_._2).map(_._1)

  private def graph(minConf: Double, withUnmerges: Boolean) = {
    val connected = sortedLinks
      .filter(_.confidence >= minConf)
      .foldLeft(Set.empty[(Int, Int)]) {
        case (set, link) =>
          val pair = (link.child, link.parent)
          val normalized = if (link.parent > link.child) pair else pair.swap
          link.`type` match {
            case LinkType.MERGE => set + normalized
            case LinkType.UNMERGE if withUnmerges => set - normalized
            case LinkType.UNMERGE => set
          }
      }
    new Graph(connected)
  }

  private def coreGraph = graph(Clusterization.defaultMinConfidence, withUnmerges = true)
  private def grayGraph = graph(minConf = 0, withUnmerges = true)
  private def rejectGraph = graph(minConf = 0, withUnmerges = false)

  val deletedIds = deletedSet
  val coreIds = {
    val result = coreGraph.getConnected(startId) -- deletedIds
    if (result.isEmpty) Set(startId)
    else result
  }
  val grayIds = grayGraph.getConnected(startId) -- (coreIds ++ deletedIds)
  val rejectIds = rejectGraph.getConnected(startId) -- (grayIds ++ coreIds ++ deletedIds)

  def distanceFor(link: ClusterLink): Int = {
    link.`type` match {
      case LinkType.MERGE if link.confidence > 0.7 => 20
      case LinkType.MERGE if link.confidence > 0.3 => 30
      case LinkType.MERGE => 40
      case LinkType.UNMERGE => 120
    }
  }

  override def toString: String = {
    s"""Deleted: $deletedIds
       |Core: $coreIds
       |Gray: $grayIds
       |Reject: $rejectIds
     """.stripMargin

  }
}
