package models

import ru.yandex.tours.backa.BackaPermalinks
import ru.yandex.tours.db.tables.HotelAmendments.DbHotelAmending
import ru.yandex.tours.db.Transaction
import ru.yandex.tours.hotels.TravelHotelBuilder
import ru.yandex.tours.model.BaseModel.Point
import ru.yandex.tours.model.{LocalizedString, MapRectangle}
import views.Utils

import scala.collection.JavaConverters._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 28.02.16
 */
class HotelClusterInfo(rawHotels: Seq[DbPartnerHotel],
                       startId: Int,
                       val graph: HotelClusterGraph,
                       val amendments: Seq[DbHotelAmending],
                       transactions: Seq[Transaction],
                       users: Seq[User],
                       backaPermalinks: BackaPermalinks) {

  private val hotelsMap = rawHotels.map(h => h.id -> h).toMap
  private val transactionsMap = transactions.map(t => t.id -> t).toMap
  private val usersMap = users.map(u => u.uid -> u).toMap

  private val (inCore, rest) = rawHotels.partition(h => graph.coreIds.contains(h.id))

  val core = TravelHotelBuilder.sortCluster(inCore.map(_.hotel)).map(ph => hotelsMap(ph.getId))
  val grayZone = rawHotels.filter(h => inGrayZone(h.id))
  val hotels = core ++ rest

  private val indexMap = hotels.zipWithIndex.map(p => (p._1.id, p._2 + 1)).toMap

  lazy val span: MapRectangle = {
    val points = (core ++ grayZone).flatMap(pointFor)
    if (points.nonEmpty) {
      MapRectangle.byPoints(points.map(p => (p.getLongitude, p.getLatitude)))
    } else {
      MapRectangle.empty
    }
  }

  lazy val result: Option[Hotel] = {
    TravelHotelBuilder.buildTravelHotel(
      core.map(_.hotel),
      Utils.hotelRatings,
      Utils.geoIdSetter,
      Utils.tree,
      amendments.map(_.amending),
      backaPermalinks
    )
      .flatMap(h => scala.util.Try(Hotel(h)).toOption)
  }

  def head: DbPartnerHotel = core.head
  def isDeleted(hotelId: Int): Boolean = graph.deletedIds.contains(hotelId)
  def inCore(hotelId: Int): Boolean = graph.coreIds.contains(hotelId)
  def inGrayZone(hotelId: Int): Boolean = graph.grayIds.contains(hotelId)
  def inRejectZone(hotelId: Int): Boolean = graph.rejectIds.contains(hotelId)

  def nameFor(hotel: DbPartnerHotel): String = {
    val stars = hotel.hotel.getRawHotel.getStars
    val starsString = if (stars > 0) " " + stars + "*" else ""
    LocalizedString.fromLangToVal(hotel.hotel.getRawHotel.getNameList.asScala).ruName + starsString
  }
  def nameFor(hotelId: Int): String = nameFor(hotelsMap(hotelId))

  def partnerFor(hotel: DbPartnerHotel): Partners.Value = Partners(hotel.hotel.getRawHotel.getPartner)
  def partnerFor(hotelId: Int): Partners.Value = partnerFor(hotelsMap(hotelId))

  def indexFor(hotelId: Int): Int = indexMap(hotelId)
  def classFor(hotelId: Int): String = {
    if (isDeleted(hotelId)) "cluster-deleted"
    else if (inCore(hotelId)) "cluster-core"
    else if (inGrayZone(hotelId)) "cluster-gray"
    else if (inRejectZone(hotelId)) "cluster-reject"
    else "cluster-other"
  }

  def pointFor(hotel: DbPartnerHotel): Option[Point] = {
    if (hotel.hotel.getRawHotel.hasPoint && !HotelsIndex.isEmptyPoint(hotel.hotel.getRawHotel.getPoint)) {
      Some(hotel.hotel.getRawHotel.getPoint)
    } else None
  }

  def transaction(id: Int): Transaction = transactionsMap(id)
  def transactionAuthor(id: Int): String = {
    val transaction = transactionsMap(id)
    val uid = transaction.author
    if (transaction.isManual) usersMap.get(uid).fold(s"Unknown (uid = $uid)")(_.login)
    else "Robot"
  }
}
