package models

import play.api.mvc.QueryStringBindable
import ru.yandex.tours.filter.Filters
import ru.yandex.tours.model.filter.hotel.GeoIdFilter
import ru.yandex.tours.model.filter.{BooleanValue, HotelFilter, IntValue, StringValue}
import ru.yandex.tours.util.spray._
import spray.http.Uri

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 02.11.15
 */
case class HotelFilters(filters: Seq[HotelFilter]) {
  def inRegion(geoId: Int): HotelFilters = copy(filters :+ new GeoIdFilter(geoId))
}

object HotelFilters {
  def empty: HotelFilters = HotelFilters(Seq.empty)
  def apply(): HotelFilters = empty

  implicit object hotelFiltersBindable extends QueryStringBindable[HotelFilters] {
    override def bind(key: String, params: Map[String, Seq[String]]): Option[Either[String, HotelFilters]] = {
      val query = Uri.Query.newBuilder
      for {
        (name, values) <- params
        value <- values
      } query += name -> value
      try {
        val filters = extract(query.result(), Filters.internalHotelFilters).filter(_.nonEmpty)
        Some(Right(HotelFilters(filters)))
      } catch {
        case ex: RuntimeException => Some(Left(ex.getMessage))
      }
    }

    override def unbind(key: String, value: HotelFilters): String = {
      val query = Uri.Query.newBuilder
      for {
        filter <- value.filters
        if filter.nonEmpty
        value <- filter.values
      } {
        value match {
          case StringValue(x) => query += filter.name -> x
          case IntValue(x) => query += filter.name -> x.toString
          case BooleanValue(x) => query += filter.name -> x.toString
          case _ =>
        }
      }
      query.result().toString()
    }
  }
}
