package models

import models.Page._
import play.api.mvc.QueryStringBindable
import ru.yandex.tours.model.util.Paging

import scala.collection.mutable

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 10.07.14 16:13
 */
case class Page(page: Int = FIRST_PAGE, size: Int = DEFAULT_SIZE) {
  def skip = (page - FIRST_PAGE) * size
  def count = size

  def prev(total: Int) = dec(total, 1)
  def next(total: Int) = add(total, 1)

  def offset(total: Int, cnt: Int) = if (cnt < 0) dec(total, -cnt) else add(total, cnt)
  private def dec(total: Int, cnt: Int) = if ((page - cnt) <= 0) None else Some(copy(page = page - cnt))
  private def add(total: Int, cnt: Int) = for (pages <- totalPages(total) if page + cnt <= pages) yield copy(page = page + cnt)

  def first(total: Int) = totalPages(total).map(_ => copy(page = FIRST_PAGE))
  def last(total: Int) = totalPages(total).map(p => copy(page = p))

  def totalPages(total: Int) = if (total > 0 && size > 0) Some(total / size) else None

  def toPaging = new Paging(page, size)

  def pages(total: Int): List[Page] = {
    val ps = new mutable.HashSet[Page]()

    first(total).foreach(ps.add)
    last(total).foreach(ps.add)
    for {
      offset <- Iterator.range(-2, 3)
      page <- this.offset(total, offset)
    } ps.add(page)

    ps.toList.sortBy(_.page)
  }
}

object Page {
  val FIRST_PAGE = 0
  val DEFAULT_SIZE = 500

  def default = new Page()

  implicit object pagerBinder extends QueryStringBindable[Page] {
    private val intBinder = implicitly[QueryStringBindable[Int]]

    override def bind(key: String, params: Map[String, Seq[String]]): Option[Either[String, Page]] = {
      Option {
        for {
          page <- intBinder.bind(key, params).getOrElse(Right(FIRST_PAGE)).right
          size <- intBinder.bind(key + ".size", params).getOrElse(Right(DEFAULT_SIZE)).right
        } yield Page(page, size)
      }
    }

    override def unbind(key: String, value: Page): String = {
      def page = intBinder.unbind(key, value.page)
      def size = intBinder.unbind(key + ".size", value.size)

      value match {
        case Page(FIRST_PAGE, DEFAULT_SIZE) => ""
        case Page(FIRST_PAGE, _) => size
        case Page(_, DEFAULT_SIZE) => page
        case Page(_, _) => page + "&" + size
      }
    }
  }
}
