package models

import com.google.common.net.UrlEscapers
import play.api.data.Form
import play.api.data.Forms._
import play.api.mvc.QueryStringBindable
import ru.yandex.tours.db.dao.HotelsDao
import ru.yandex.tours.db.dao.HotelsDao.{IsNew, OnlyPartner, WithUrl}
import ru.yandex.tours.model.hotels.Partners._
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.parsing.IntValue
import spray.http.Uri

import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 13.04.16
 */
case class PartnerHotelRequest(url: Option[String],
                               partner: Option[Partner],
                               isNew: Option[Boolean]) {

  def toQuery: Seq[HotelsDao.Query] = Seq(
    url.map(WithUrl),
    partner.map(OnlyPartner),
    isNew.map(IsNew)
  ).flatten
}

object PartnerHotelRequest extends Logging {

  def empty: PartnerHotelRequest = PartnerHotelRequest(None, None, None)

  implicit object partnerHotelRequestBindable extends QueryStringBindable[PartnerHotelRequest] {
    override def bind(key: String, params: Map[String, Seq[String]]): Option[Either[String, PartnerHotelRequest]] = {
      try {
        val url = params.get("url").flatMap(_.headOption)
        val partner = params.get("partner").flatMap(_.headOption).map(parsePartner)
        val isNew = params.get("is_new").flatMap(_.headOption).map(_.toBoolean)

        Some(Right(PartnerHotelRequest(url, partner, isNew)))
      } catch {
        case NonFatal(t) =>
          Some(Left(t.toString))
      }
    }

    override def unbind(key: String, value: PartnerHotelRequest): String = {
      val b = Uri.Query.newBuilder

      value.url.foreach { b += "url" -> _ }
      value.partner.foreach { b += "partner" -> _.toString }
      value.isNew.foreach { b += "is_new" -> _.toString }

      b.result().toString()
    }
  }

  val form = Form(mapping(
    "url" -> optional(text.transform[String](unifyPartnerUrl, identity)),
    "partner" -> optional(text.transform[Partner](parsePartner, _.toString)),
    "is_new" -> optional(boolean)
  )(PartnerHotelRequest.apply)(PartnerHotelRequest.unapply))

  def parsePartner(value: String): Partners.Value = value match {
    case IntValue(id) => Partners.apply(id)
    case name => Partners.withName(name)
  }

  import ru.yandex.tours.util.naming.HotelNameUtils._

  def unifyPartnerUrl(url: String): String = url match {
    case Booking(country, name) =>
      val withoutCountry = name.split("\\.").head
      s"https://www.booking.com/hotel/$country/$withoutCountry.html"
    case Ostrovok(name) =>
      s"https://ostrovok.ru/rooms/$name/"
    case Oktogo(name) =>
      s"http://oktogo.ru/$name/"
    case LevelTravel(id, name) =>
      val escapedName =
        if ("[а-яА-Я]+".r.findFirstIn(name).nonEmpty) UrlEscapers.urlPathSegmentEscaper().escape(name)
        else name
      s"https://level.travel/hotels/$id-$escapedName"
    case Hotels101(city, name) =>
      s"https://www.101hotels.ru/main/cities/$city/$name.html"
    case YandexMaps(name, id) ⇒
      s"https://maps.yandex.ru/org/$id"
    case matched =>
      log.info(s"[$url] not matched in unification")
      url
  }
}