package models

import play.api.mvc.QueryStringBindable
import ru.yandex.tours.model.BaseModel.Currency
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.search._
import spray.http.Uri.Query

import scala.util.{Failure, Success, Try}

case class SearchRequestWrapper(searchRequest: ExtendedBaseRequest)

object SearchRequestWrapper {
  implicit object searchRequestBinder extends QueryStringBindable[SearchRequestWrapper] {
    private def getSingle(prefix: String, key: String, params: Map[String, Seq[String]]): Option[String] = {
      for {
        result <- params.get(prefix + key)
        head <- result.headOption
      } yield head
    }

    override def bind(key: String, params: Map[String, Seq[String]]): Option[Either[String, SearchRequestWrapper]] = {
      val prefix = if (key.isEmpty) "" else s"$key."
      Try {
        {
          for {
            rawDate <- getSingle(prefix, "when", params)
            when = org.joda.time.LocalDate.parse(rawDate)
            to = getSingle(prefix, "to", params).map(_.toInt).getOrElse(0)
            from = getSingle(prefix, "from", params).map(_.toInt).getOrElse(0)
            nights <- getSingle(prefix, "nights", params).map(_.toInt)
            rawAges <- params.get(prefix + "ages")
            ages = rawAges.flatMap(_.split(",")).map(_.toInt)
          } yield {
            val whenFlex = getSingle(prefix, "when_flex", params).map(_.toBoolean).getOrElse(false)
            val nightsFlex = getSingle(prefix, "nights_flex", params).map(_.toBoolean).getOrElse(false)
            val currency = getSingle(prefix, "currency", params).map(Currency.valueOf).getOrElse(Currency.RUB)
            val lang = getSingle(prefix, "lang", params).map(Languages.withName).getOrElse(Languages.ru)
            val filter = SearchFilter.parse(getSingle(prefix, SearchFilter.name, params).getOrElse(""))
            HotelSearchRequest(from, to, nights, when, ages, whenFlex, nightsFlex, currency, lang, filter)
          }
        }.map { hsr =>
          getSingle(prefix, "hotel_id", params) match {
            case Some(hotelId) => ExtendedOfferSearchRequest(OfferSearchRequest(hsr, hotelId.toInt))
            case None => ExtendedHotelSearchRequest(hsr)
          }
        }
      } match {
        case Success(Some(req)) => Some(Right(SearchRequestWrapper(req)))
        case Success(None) => if (key.isEmpty) None else bind("", params)
        case Failure(e) => Some(Left(e.getMessage))
      }
    }

    override def unbind(key: String, value: SearchRequestWrapper): String = {
      val prefix = if (key.isEmpty) "" else key + "."
      val hotelParam = value match {
        case SearchRequestWrapper(ExtendedOfferSearchRequest(OfferSearchRequest(_, hotelId))) =>
          Seq(s"hotel_id" -> hotelId.toString)
        case _ => Seq.empty[(String, String)]
      }

      val request = value.searchRequest.hotelRequest
      val params = hotelParam ++ Seq(
        "to" -> request.to.toString,
        "from" -> request.from.toString,
        "nights" -> request.nights.toString,
        "when" -> request.when.toString,
        "when_flex" -> request.flexWhen.toString,
        "nights_flex" -> request.flexNights.toString,
        "currency" → request.currency.name(),
        "lang" → request.lang.toString,
        "filter" -> request.filter.toString
      ) ++ request.ages.map(age => "ages" -> age.toString)

      Query(params.map {
        case (k, v) => (prefix + k) -> v
      }: _*).toString()
    }
  }
}
