package ru.yandex.tours.db.geomapping

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.mapping.GeoMappingShort
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners._

/**
  * Created by asoboll on 29.12.15.
  */
case class RegionGeoMappings(geoId: Int, banStatus: BanStatus,
                             countryMap: Map[Partner, String],
                             cityMap: Map[Partner, String],
                             departureMap: Map[Partner, String],
                             airportMap: Map[Partner, String]) {
  def keySet: Set[Partner] = countryMap.keySet ++ cityMap.keySet ++ departureMap.keySet ++ airportMap.keySet
  val keys = keySet.toSeq.sorted

  private def takeMap(dataType: DataType) = {
    dataType match {
      case DataTypes.countries => this.countryMap
      case DataTypes.cities => this.cityMap
      case DataTypes.departures => this.departureMap
      case DataTypes.airports => this.airportMap
      case _ => Map(): Map[Partner, String]
    }
  }

  def diffPatch(that: RegionGeoMappings, dataType: DataType): Set[(Command, GeoMappingShort)] =
    GeoMappingRecordProcessor.diff(this.geoId, this.takeMap(dataType), that.takeMap(dataType))

  def banPatch(that: RegionGeoMappings): Option[(Command, GeoMappingShort)] = {
    (this.banStatus, that.banStatus) match {
      case (NotBanned, BannedThis) => Some(UnBanned: Command, GeoMappingShort(Partners.unknown, geoId, "undefined"))
      case (BannedThis, NotBanned) => Some(Banned: Command, GeoMappingShort(Partners.unknown, geoId, "undefined"))
      case _ => None
    }
  }
}

/**
  * BanStatus enables and disables region (with all children regions) geoId -> partnerIds mapping
  * BannedThis means that region is banned itself
  * BannedAt some id means that some parent of region is banned
  * NotBanned means that region is not banned
  *
  */
sealed trait BanStatus {
  def bannedAt: Int = this match {
    case BannedAt(x) => x
    case _ => 0
  }

  def isBanned: Boolean = this != NotBanned
}

case object BannedThis extends BanStatus
case object NotBanned extends BanStatus
case class BannedAt(id: Int) extends BanStatus