package ru.yandex.tours.db.searchsettings

import ru.yandex.tours.db.{DBWrapper, SearchSettings}
import ru.yandex.tours.geo.base.Region
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.model.search.settings.{SearchSettingsRecord, Parameters => SCParameters}
import ru.yandex.tours.model.search.{SearchOptions, SearchType}
import ru.yandex.tours.search.settings.RegionSearchSettings
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang.Futures._
import slick.driver.MySQLDriver.api._

import scala.concurrent.{ExecutionContext, Future}

/**
  * Created by Evgeny Zhoga on 02.02.16.
  */
class DbSearchSettings(db: DBWrapper, tree: Tree)(implicit ex: ExecutionContext) extends Logging {

  private def getSearchSettings(ids: List[Int]): Future[Seq[SearchSettingsRecord]] =
    db.run(SearchSettings.table.filter(_.regionId inSet ids).result)

  def getRegionSearchSettings(region: Region): Future[RegionSearchSettings] = {
    val path = tree.pathToRoot(region.id)

    val idPath = path.map(_.id)
    getSearchSettings(idPath).map { res =>
      val params = res.flatMap { sc =>
        sc.parameters.searchByType.toSeq.map {
          case (searchType, searchOption) => (sc.regionId, searchType) -> searchOption
        }
      }.toMap
      RegionSearchSettings(region, params, getIsForcedSearch(region, res))
    }
  }

  def updateSearchSettings(regionId: Int, searchType: SearchType.SearchType,
                           searchOption: SearchOptions.SearchOption): Future[Int] = {
    getSearchSettings(List(regionId)).flatMap { mappingsOpt =>
      val mappingOpt = mappingsOpt.headOption
      val record = mappingOpt.fold {
        SearchSettingsRecord(regionId, SCParameters(Map(searchType -> searchOption), None))
      } {
        mapping => mapping + (searchType -> searchOption)
      }
      val q = SearchSettings.table.insertOrUpdate(record)
      db.run(q)
    }
  }

  def deleteSearchSettings(regionId: Int, searchType: SearchType.SearchType): Future[Unit] = {
    getSearchSettings(List(regionId)).map(_.headOption).flatMap {
      case Some(mapping) =>
        val params = mapping.parameters.searchByType - searchType
        val updated = mapping.copy(parameters = mapping.parameters.copy(searchByType = params))
        val query = SearchSettings.table.insertOrUpdate(updated)
        db.run(query).toUnit
      case v =>
        Future.successful(())
    }
  }

  private def getIsForcedSearch(region: Region, res: Seq[SearchSettingsRecord]): Boolean = {
    res.find(_.regionId == region.id).flatMap(_.parameters.forcedSearchable).getOrElse(false)
  }
}